<?php
namespace App\Http\Controllers;

use App\Models\ActivityLog;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class ActivityLogController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:activity-log.view', ['only' => ['index', 'show', 'stats']]);
        $this->middleware('permission:activity-log.export', ['only' => ['export']]);
        $this->middleware('permission:activity-log.clear', ['only' => ['clear']]);
    }
    public function index(Request $request)
    {
        $query = ActivityLog::with('user')->latest('waktu');

        // Filter berdasarkan parameter
        if ($request->filled('status')) {
            $query->byStatus($request->status);
        }

        if ($request->filled('module')) {
            $query->byModule($request->module);
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $query->byDateRange($request->date_from, $request->date_to);
        }

        if ($request->filled('user_name')) {
            $query->where('user_name', 'like', '%' . $request->user_name . '%');
        }

        if ($request->filled('aktivitas')) {
            $query->where('aktivitas', 'like', '%' . $request->aktivitas . '%');
        }

        if ($request->filled('ip_address')) {
            $query->where('ip_address', $request->ip_address);
        }

        $activities = $query->paginate(50);

        // Return JSON untuk API request
        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $activities->items(),
                'pagination' => [
                    'current_page' => $activities->currentPage(),
                    'total_pages' => $activities->lastPage(),
                    'total_items' => $activities->total(),
                    'per_page' => $activities->perPage(),
                ]
            ]);
        }
        $title = 'Activity Logs';
        // Return view untuk web request
        return view('admin.activity-log.index', compact('activities','title'));
    }

    public function show(ActivityLog $activityLog)
    {
        return response()->json([
            'success' => true,
            'data' => $activityLog->load('user')
        ]);
    }

    public function stats(Request $request)
    {
        $dateFrom = $request->get('date_from', now()->subDays(30));
        $dateTo = $request->get('date_to', now());

        $stats = [
            'total_activities' => ActivityLog::byDateRange($dateFrom, $dateTo)->count(),
            'successful_activities' => ActivityLog::byDateRange($dateFrom, $dateTo)->byStatus('Berhasil')->count(),
            'failed_activities' => ActivityLog::byDateRange($dateFrom, $dateTo)->byStatus('Gagal')->count(),
            'warning_activities' => ActivityLog::byDateRange($dateFrom, $dateTo)->byStatus('Peringatan')->count(),
            'unique_users' => ActivityLog::byDateRange($dateFrom, $dateTo)->distinct('user_id')->count('user_id'),
            'top_activities' => ActivityLog::byDateRange($dateFrom, $dateTo)
                ->selectRaw('aktivitas, COUNT(*) as total')
                ->groupBy('aktivitas')
                ->orderByDesc('total')
                ->limit(10)
                ->get(),
            'top_modules' => ActivityLog::byDateRange($dateFrom, $dateTo)
                ->selectRaw('module, COUNT(*) as total')
                ->groupBy('module')
                ->orderByDesc('total')
                ->limit(10)
                ->get(),
            'activities_by_hour' => ActivityLog::byDateRange($dateFrom, $dateTo)
                ->selectRaw('HOUR(waktu) as hour, COUNT(*) as total')
                ->groupBy('hour')
                ->orderBy('hour')
                ->get(),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }

    public function export(Request $request)
    {
        $query = ActivityLog::with('user')->latest('waktu');

        // Apply same filters as index
        if ($request->filled('status')) {
            $query->byStatus($request->status);
        }

        if ($request->filled('module')) {
            $query->byModule($request->module);
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $query->byDateRange($request->date_from, $request->date_to);
        }

        $activities = $query->get();

        $filename = 'activity_logs_' . now()->format('Y-m-d_H-i-s') . '.csv';

        $headers = [
            'Content-type' => 'text/csv',
            'Content-Disposition' => "attachment; filename={$filename}",
        ];

        $callback = function() use ($activities) {
            $file = fopen('php://output', 'w');

            // Header CSV
            fputcsv($file, [
                'Waktu', 'User', 'Role', 'Aktivitas', 'Deskripsi',
                'Module', 'Status', 'IP Address'
            ]);

            // Data CSV
            foreach ($activities as $activity) {
                fputcsv($file, [
                    $activity->waktu->format('Y-m-d H:i:s'),
                    $activity->user_name,
                    $activity->user_role,
                    $activity->aktivitas,
                    $activity->deskripsi,
                    $activity->module,
                    $activity->status,
                    $activity->ip_address,
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function clear(Request $request)
    {
        $request->validate([
            'days' => 'required|integer|min:1',
        ]);

        $cutoffDate = now()->subDays($request->days);
        $deletedCount = ActivityLog::where('waktu', '<', $cutoffDate)->delete();

        // Log pembersihan
        ActivityLog::logActivity(
            'Clear Activity Logs',
            "Membersihkan {$deletedCount} log aktivitas yang lebih lama dari {$request->days} hari",
            'System',
            'Berhasil',
            ['deleted_count' => $deletedCount, 'days' => $request->days]
        );

        return response()->json([
            'success' => true,
            'message' => "Berhasil menghapus {$deletedCount} log aktivitas",
            'deleted_count' => $deletedCount
        ]);
    }
}
