<?php

namespace App\Http\Controllers;

use App\Models\Kecamatan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class KecamatanController extends Controller
{

    /**
     * Display a listing of the resource.
     *
     *
     * @return \Illuminate\Http\Response
     */
    public function __construct()
    {
        $this->middleware('permission:kecamatan.view', ['only' => ['index', 'show']]);
        $this->middleware('permission:kecamatan.create', ['only' => ['create', 'store']]);
        $this->middleware('permission:kecamatan.edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:kecamatan.delete', ['only' => ['destroy']]);
    }
    public function index(Request $request)
    {
        $query = Kecamatan::query();

        // Filter pencarian (kode atau nama)
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('kode', 'like', "%{$search}%")
                    ->orWhere('nama', 'like', "%{$search}%");
            });
        }

        // Filter status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $kecamatans = $query->paginate(10)->appends($request->query());

        return view('admin.kecamatan.index', compact('kecamatans'));
    }


    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.kecamatan.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response

     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Kecamatan  $kecamatan
     * @return \Illuminate\Http\Response
     */

        public function store(Request $request)
        {
            $request->validate([
                'kode_kecamatan' => 'required|unique:kecamatan,kode_kecamatan',
                'nama_kecamatan' => 'required',
                'alamat_kantor_kecamatan' => 'required',
                'kode_pos' => 'required|numeric',
                'nama_camat' => 'required',
                'nip' => 'nullable',
                'golongan' => 'nullable',
                'pangkat' => 'nullable',
                'luas_wilayah' => 'required|numeric',
                'jumlah_desa' => 'required|integer',
                'status' => 'required|in:Aktif,Nonaktif',
                'warna' => 'required',
                'geojson' => 'nullable|file|mimes:json|max:5048', // Validasi file GeoJSON
            ]);

            $data = $request->all();

            // Handle file GeoJSON jika ada
            if ($request->hasFile('geojson')) {
                $geojsonFile = $request->file('geojson');
                $fileName = time() . '_' . $geojsonFile->getClientOriginalName();
                $geojsonContent = file_get_contents($geojsonFile->getRealPath());
                $data['geojson'] = json_encode(json_decode($geojsonContent)); // Simpan konten JSON
                $data['geojson_file'] = $fileName; // Simpan nama file
                Storage::putFileAs('public/geojson', $geojsonFile, $fileName); // Simpan file ke storage
            }

            Kecamatan::create($data);

            return redirect()->route('kecamatan.index')->with('success', 'Kecamatan berhasil ditambahkan.');
        }
    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Kecamatan  $kecamatan
     * @return \Illuminate\Http\Response
     */
    public function show(Kecamatan $kecamatan)
    {
        return view('admin.kecamatan.show', compact('kecamatan'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Kecamatan  $kecamatan
     * @return \Illuminate\Http\Response
     */
    public function edit(Kecamatan $kecamatan)
    {
        return view('admin.kecamatan.edit', compact('kecamatan'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Kecamatan  $kecamatan
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Kecamatan $kecamatan)
    {
        $request->validate([
            'kode_kecamatan' => 'required|unique:kecamatan,kode_kecamatan,' . $kecamatan->id,
            'nama_kecamatan' => 'required',
            'alamat_kantor_kecamatan' => 'required',
            'kode_pos' => 'required|numeric',
            'nama_camat' => 'required',
            'nip' => 'nullable',
            'golongan' => 'nullable',
            'pangkat' => 'nullable',
            'luas_wilayah' => 'required|numeric',
            'jumlah_desa' => 'required|integer',
            'status' => 'required|in:Aktif,Nonaktif',
            'warna' => 'required',
            'geojson' => 'nullable|file|mimes:json|max:5048', // Validasi file GeoJSON
        ]);

        $data = $request->all();

        // Handle file GeoJSON jika ada
        if ($request->hasFile('geojson')) {
            // Hapus file lama jika ada
            if ($kecamatan->geojson_file && Storage::exists('public/geojson/' . $kecamatan->geojson_file)) {
                Storage::delete('public/geojson/' . $kecamatan->geojson_file);
            }

            $geojsonFile = $request->file('geojson');
            $fileName = time() . '_' . $geojsonFile->getClientOriginalName();
            $geojsonContent = file_get_contents($geojsonFile->getRealPath());
            $data['geojson'] = json_encode(json_decode($geojsonContent)); // Simpan konten JSON
            $data['geojson_file'] = $fileName; // Simpan nama file baru
            Storage::putFileAs('public/geojson', $geojsonFile, $fileName); // Simpan file baru
        } elseif ($request->input('geojson') === null && $request->has('remove_geojson')) {
            // Hapus file dan konten jika ada permintaan penghapusan
            if ($kecamatan->geojson_file && Storage::exists('public/geojson/' . $kecamatan->geojson_file)) {
                Storage::delete('public/geojson/' . $kecamatan->geojson_file);
            }
            $data['geojson'] = null;
            $data['geojson_file'] = null;
        }

        $kecamatan->update($data);

        return redirect()->route('kecamatan.index')->with('success', 'Kecamatan berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Kecamatan  $kecamatan
     * @return \Illuminate\Http\Response
     */
    public function destroy(Kecamatan $kecamatan)
    {
        $kecamatan->delete();

        return redirect()->route('kecamatan.index')->with('success', 'Kecamatan berhasil dihapus.');
    }
}
