<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class RoleController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:role.view', ['only' => ['index', 'show']]);
        $this->middleware('permission:role.create', ['only' => ['create', 'store']]);
        $this->middleware('permission:role.edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:role.delete', ['only' => ['destroy']]);
        $this->middleware('permission:role.assign-permissions', ['only' => ['permissions', 'updatePermissions', 'syncPermissions']]);
    }

    public function index()
    {
        $roles = Role::with(['permissions', 'users'])
            ->orderBy('name')
            ->paginate(10);

        return view('admin.roles.index', compact('roles'));
    }

    public function create()
    {
        // Group permissions by extracting the prefix before the dot
        $permissions = Permission::all();
        $groupedPermissions = $permissions->groupBy(function($permission) {
            $parts = explode('.', $permission->name);
            return $parts[0] ?? 'Other';
        });

        return view('admin.roles.create', compact('groupedPermissions'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => [
                'required',
                'string',
                'max:255',
                'regex:/^[a-z0-9_]+$/',
                Rule::unique('roles', 'name')
            ],
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id'
        ], [
            'name.required' => 'Role name is required',
            'name.regex' => 'Role name must contain only lowercase letters, numbers, and underscores',
            'name.unique' => 'This role name already exists',
            'permissions.*.exists' => 'One or more selected permissions are invalid'
        ]);

        DB::beginTransaction();

        try {
            // Create role
            $role = Role::create([
                'name' => $validated['name'],
                'guard_name' => 'web'
            ]);

            // Assign permissions if any
            if (!empty($validated['permissions'])) {
                // Get permission objects by IDs
                $permissionIds = $validated['permissions'];
                $permissions = Permission::whereIn('id', $permissionIds)->get();

                // Sync permissions using permission names or objects
                $role->syncPermissions($permissions);
            }

            DB::commit();

            $permissionCount = count($validated['permissions'] ?? []);

            return redirect()->route('roles.index')
                ->with('success', "Role '{$role->name}' has been created successfully with {$permissionCount} permission(s).");

        } catch (\Exception $e) {
            DB::rollback();

            \Log::error('Role creation failed: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            return back()
                ->withInput()
                ->with('error', 'Failed to create role: ' . $e->getMessage());
        }
    }

    public function show(Role $role)
    {
        $role->load('permissions', 'users');
        return view('admin.roles.show', compact('role'));
    }

    public function edit(Role $role)
    {
        $permissions = Permission::all();
        $groupedPermissions = $permissions->groupBy(function($permission) {
            $parts = explode('.', $permission->name);
            return $parts[0] ?? 'Other';
        });

        $rolePermissions = $role->permissions->pluck('id')->toArray();

        return view('admin.roles.edit', compact('role', 'groupedPermissions', 'rolePermissions', 'permissions'));
    }

    public function update(Request $request, Role $role)
    {
        // Prevent editing superadmin role
        if ($role->name === 'superadmin' && !auth()->user()->hasRole('superadmin')) {
            return back()->with('error', 'You cannot edit the superadmin role.');
        }

        $validated = $request->validate([
            'name' => [
                'required',
                'string',
                'max:255',
                'regex:/^[a-z0-9_]+$/',
                Rule::unique('roles', 'name')->ignore($role->id)
            ],
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id'
        ], [
            'name.required' => 'Role name is required',
            'name.regex' => 'Role name must contain only lowercase letters, numbers, and underscores',
            'name.unique' => 'This role name already exists',
            'permissions.*.exists' => 'One or more selected permissions are invalid'
        ]);

        DB::beginTransaction();

        try {
            // Update role name
            $role->update([
                'name' => $validated['name'],
            ]);

            // Sync permissions
            if (isset($validated['permissions']) && !empty($validated['permissions'])) {
                // Get permission objects by IDs
                $permissionIds = $validated['permissions'];
                $permissions = Permission::whereIn('id', $permissionIds)->get();

                // Sync permissions using permission objects
                $role->syncPermissions($permissions);
            } else {
                // If no permissions selected, remove all
                $role->syncPermissions([]);
            }

            // Clear permission cache
            app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

            DB::commit();

            return redirect()->route('roles.index')
                ->with('success', "Role '{$role->name}' has been updated successfully.");

        } catch (\Exception $e) {
            DB::rollback();

            \Log::error('Role update failed: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            return back()
                ->withInput()
                ->with('error', 'Failed to update role: ' . $e->getMessage());
        }
    }

    public function destroy(Role $role)
    {
        // Prevent deleting superadmin role
        if ($role->name === 'superadmin') {
            return back()->with('error', 'Cannot delete the superadmin role.');
        }

        // Check if role has users
        if ($role->users()->count() > 0) {
            return back()->with('error', "Cannot delete role '{$role->name}' because it is assigned to " . $role->users()->count() . " user(s). Please reassign them first.");
        }

        DB::beginTransaction();

        try {
            $roleName = $role->name;

            // Remove all permissions before deleting
            $role->syncPermissions([]);

            // Delete the role
            $role->delete();

            // Clear permission cache
            app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

            DB::commit();

            return redirect()->route('roles.index')
                ->with('success', "Role '{$roleName}' has been deleted successfully.");

        } catch (\Exception $e) {
            DB::rollback();

            \Log::error('Role deletion failed: ' . $e->getMessage());

            return back()->with('error', 'Failed to delete role: ' . $e->getMessage());
        }
    }
}
