<?php

namespace App\Http\Controllers;

use App\Models\Rtrw;
use App\Models\Kecamatan;
use App\Models\Kampung;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class RtrwController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function __construct()
    {
        $this->middleware('permission:rtrw.view', ['only' => ['index', 'show', 'apiAll']]);
        $this->middleware('permission:rtrw.create', ['only' => ['create', 'store']]);
        $this->middleware('permission:rtrw.edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:rtrw.delete', ['only' => ['destroy']]);
    }
    public function index(Request $request)
    {
        $query = Rtrw::query();

        // Filter pencarian (kode atau RTRW)
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('nama_rtrw', 'like', "%{$search}%")
                ->orWhere('warna', 'like', "%{$search}%");

            });
        }

        // Filter status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $rtrws = $query->paginate(10)->appends($request->query());

        return view('admin.rtrw.index', compact('rtrws'));
    }


    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $kecamatans = Kecamatan::all();
        $kampungs = Kampung::all();
        return view('admin.rtrw.create', compact('kecamatans', 'kampungs'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
{
    $request->validate([
        'nama_rtrw' => 'required|string|max:255',
        'warna' => 'required|string|max:7',
        'geojson' => 'nullable|file|max:2048', // 2MB max
    ]);

    $data = [
        'nama_rtrw' => $request->nama_rtrw,
        'warna' => $request->warna,
    ];

    // Handle GeoJSON file upload
    if ($request->hasFile('geojson')) {
        $file = $request->file('geojson');

        // Validate JSON content
        $content = file_get_contents($file->getRealPath());
        $geojsonData = json_decode($content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return back()
                ->withErrors(['geojson' => 'File JSON tidak valid'])
                ->withInput();
        }

        // Validate GeoJSON structure
        if (!isset($geojsonData['type'])) {
            return back()
                ->withErrors(['geojson' => 'Format GeoJSON tidak valid: property "type" tidak ditemukan'])
                ->withInput();
        }

        // Store new file
        $fileName = time() . '_' . $file->getClientOriginalName();
        $file->storeAs('geojson', $fileName, 'public');

        $data['geojson_file'] = $fileName;
        $data['geojson'] = $content; // Store JSON content in database
    }

    Rtrw::create($data);

    return redirect()
        ->route('rtrw.index')
        ->with('success', 'Data RTRW berhasil ditambahkan');
}


    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Rtrw  $rtrw
     * @return \Illuminate\Http\Response
     */
    public function show(Rtrw $rtrw)
    {
        return view('admin.rtrw.show', compact('rtrw'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Rtrw  $rtrw
     * @return \Illuminate\Http\Response
     */
    public function edit(Rtrw $rtrw)
    {
        $kecamatans = Kecamatan::all();
        $kampungs = Kampung::all();
        return view('admin.rtrw.edit', compact('rtrw', 'kecamatans', 'kampungs'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Rtrw  $rtrw
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Rtrw $rtrw)
    {
        $request->validate([
            'nama_rtrw' => 'required|string|max:255',
            'warna' => 'required|string|max:7',
            'geojson' => 'nullable|file|max:2048', // 2MB max
        ]);

        $data = [
            'nama_rtrw' => $request->nama_rtrw,
            'warna' => $request->warna,
        ];

        // Handle GeoJSON file upload
        if ($request->hasFile('geojson')) {
            $file = $request->file('geojson');

            // Validate JSON content
            $content = file_get_contents($file->getRealPath());
            $geojsonData = json_decode($content, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                return back()
                    ->withErrors(['geojson' => 'File JSON tidak valid'])
                    ->withInput();
            }

            // Validate GeoJSON structure
            if (!isset($geojsonData['type'])) {
                return back()
                    ->withErrors(['geojson' => 'Format GeoJSON tidak valid: property "type" tidak ditemukan'])
                    ->withInput();
            }

            // Delete old file if exists
            if ($rtrw->geojson_file && Storage::disk('public')->exists('geojson/' . $rtrw->geojson_file)) {
                Storage::disk('public')->delete('geojson/' . $rtrw->geojson_file);
            }

            // Store new file
            $fileName = time() . '_' . $file->getClientOriginalName();
            $file->storeAs('geojson', $fileName, 'public');

            $data['geojson_file'] = $fileName;
            $data['geojson'] = $content; // Store JSON content in database
        }

        $rtrw->update($data);

        return redirect()
            ->route('rtrw.index')
            ->with('success', 'Data RTRW berhasil diperbarui');
    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Rtrw  $rtrw
     * @return \Illuminate\Http\Response
     */
    public function destroy(Rtrw $rtrw)
    {
        $rtrw->delete();

        return redirect()->route('rtrw.index')->with('success', 'RTRW berhasil dihapus.');
    }
     public function apiAll()
    {
        try {
            $rtrws = Rtrw::select('id', 'nama_rtrw', 'warna', 'geojson', 'geojson_file')
                ->get()
                ->map(function($rtrw) {
                    // Parse geojson jika masih string
                    if (is_string($rtrw->geojson)) {
                        try {
                            $rtrw->geojson = json_decode($rtrw->geojson, true);
                        } catch (\Exception $e) {
                            $rtrw->geojson = null;
                        }
                    }

                    return [
                        'id' => $rtrw->id,
                        'nama_rtrw' => $rtrw->nama_rtrw,
                        'warna' => $rtrw->warna,
                        'geojson' => $rtrw->geojson,
                        'geojson_file' => $rtrw->geojson_file
                    ];
                })
                ->filter(function($rtrw) {
                    // Filter yang punya geojson valid
                    return !empty($rtrw['geojson']);
                })
                ->values(); // Reset array keys

            return response()->json($rtrws);

        } catch (\Exception $e) {
            \Log::error('Error fetching RTRW data for API: ' . $e->getMessage());

            return response()->json([
                'error' => 'Gagal mengambil data RTRW',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
