<?php

namespace App\Http\Controllers;

use App\Models\User;
use Spatie\Permission\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:user.view', ['only' => ['index', 'show']]);
        $this->middleware('permission:user.create', ['only' => ['create', 'store']]);
        $this->middleware('permission:user.edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:user.delete', ['only' => ['destroy']]);

        // Profile routes accessible by all authenticated users
        $this->middleware('auth', ['only' => ['showEditProfile', 'editProfile']]);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $users = User::with(['roles', 'primaryRole'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        $title = 'Users Management';

        return view('admin.users.index', compact('users', 'title'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $roles = Role::orderBy('name')->get();
        $title = 'Create User';

        return view('admin.users.create', compact('roles', 'title'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'username' => 'required|string|max:255|unique:users|alpha_dash',
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'roles' => 'required|array|min:1',
            'roles.*' => 'exists:roles,id',
            'primary_role_id' => [
                'required',
                'exists:roles,id',
                Rule::in($request->roles ?? [])
            ],
            'profile_photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ], [
            'username.required' => 'Username is required',
            'username.unique' => 'Username already exists',
            'username.alpha_dash' => 'Username can only contain letters, numbers, dashes and underscores',
            'name.required' => 'Name is required',
            'email.required' => 'Email is required',
            'email.unique' => 'Email already exists',
            'password.required' => 'Password is required',
            'password.min' => 'Password must be at least 8 characters',
            'password.confirmed' => 'Password confirmation does not match',
            'roles.required' => 'Please select at least one role',
            'roles.min' => 'Please select at least one role',
            'primary_role_id.required' => 'Primary role is required',
            'primary_role_id.in' => 'Primary role must be one of the selected roles',
        ]);

        DB::beginTransaction();
        try {
            // Handle profile photo upload
            $photoPath = null;
            if ($request->hasFile('profile_photo')) {
                $file = $request->file('profile_photo');
                $filename = time() . '_' . $file->getClientOriginalName();
                $photoPath = $file->storeAs('profile_photos', $filename, 'public');
            }

            // Create user
            $user = User::create([
                'username' => $validated['username'],
                'name' => $validated['name'],
                'email' => $validated['email'],
                'password' => Hash::make($validated['password']),
                'primary_role_id' => $validated['primary_role_id'],
                'profile_photo' => $photoPath,
            ]);

            // Get role objects by IDs
            $roleIds = $validated['roles'];
            $roles = Role::whereIn('id', $roleIds)->get();

            // Assign roles using Spatie - using role objects
            $user->syncRoles($roles);

            // Clear permission cache
            app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

            DB::commit();

            \Log::info('User created successfully', [
                'user_id' => $user->id,
                'username' => $user->username,
                'roles' => $user->roles->pluck('name')
            ]);

            return redirect()->route('users.index')
                ->with('success', "User '{$user->name}' created successfully with " . count($roleIds) . " role(s).");

        } catch (\Exception $e) {
            DB::rollback();

            \Log::error('Failed to create user', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Delete uploaded photo if error occurs
            if (isset($photoPath) && $photoPath) {
                Storage::disk('public')->delete($photoPath);
            }

            return back()->withInput()
                ->with('error', 'Failed to create user: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        $user->load(['roles', 'primaryRole']);
        $permissions = $user->getAllPermissions();
        $roles = Role::orderBy('name')->get();
        $title = 'User Details';
        $userRoles = $user->roles->pluck('id')->toArray();

        return view('admin.users.show', compact('user', 'permissions', 'roles', 'title', 'userRoles'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $user->load('roles');
        $roles = Role::orderBy('name')->get();
        $userRoles = $user->roles->pluck('id')->toArray();
        $title = 'Edit User';

        return view('admin.users.edit', compact('user', 'roles', 'userRoles', 'title'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'username' => [
                'required',
                'string',
                'max:255',
                'alpha_dash',
                Rule::unique('users')->ignore($user->id)
            ],
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('users')->ignore($user->id)
            ],
            'password' => 'nullable|string|min:8|confirmed',
            'roles' => 'required|array|min:1',
            'roles.*' => 'exists:roles,id',
            'primary_role_id' => [
                'required',
                'exists:roles,id',
                Rule::in($request->roles ?? [])
            ],
            'profile_photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ], [
            'username.required' => 'Username is required',
            'username.unique' => 'Username already exists',
            'username.alpha_dash' => 'Username can only contain letters, numbers, dashes and underscores',
            'name.required' => 'Name is required',
            'email.required' => 'Email is required',
            'email.unique' => 'Email already exists',
            'password.min' => 'Password must be at least 8 characters',
            'password.confirmed' => 'Password confirmation does not match',
            'roles.required' => 'Please select at least one role',
            'roles.min' => 'Please select at least one role',
            'primary_role_id.required' => 'Primary role is required',
            'primary_role_id.in' => 'Primary role must be one of the selected roles',
        ]);

        // Prevent removing superadmin role from superadmin user
        $superadminRole = Role::where('name', 'superadmin')->first();
        if ($superadminRole && $user->hasRole('superadmin')) {
            if (!in_array($superadminRole->id, $request->roles)) {
                if ($user->id === auth()->id()) {
                    return back()->with('error', 'You cannot remove superadmin role from yourself.');
                }

                // Only superadmin can remove superadmin role from others
                if (!auth()->user()->hasRole('superadmin')) {
                    return back()->with('error', 'Only superadmin can modify superadmin roles.');
                }
            }
        }

        DB::beginTransaction();
        try {
            $updateData = [
                'username' => $validated['username'],
                'name' => $validated['name'],
                'email' => $validated['email'],
                'primary_role_id' => $validated['primary_role_id'],
            ];

            // Update password if provided
            if ($request->filled('password')) {
                $updateData['password'] = Hash::make($validated['password']);
            }

            // Handle profile photo upload
            if ($request->hasFile('profile_photo')) {
                // Delete old photo
                if ($user->profile_photo) {
                    Storage::disk('public')->delete($user->profile_photo);
                }

                $file = $request->file('profile_photo');
                $filename = time() . '_' . $file->getClientOriginalName();
                $updateData['profile_photo'] = $file->storeAs('profile_photos', $filename, 'public');
            }

            // Update user data
            $user->update($updateData);

            // Get role objects by IDs
            $roleIds = $validated['roles'];
            $roles = Role::whereIn('id', $roleIds)->get();

            // Sync roles using Spatie - using role objects
            $user->syncRoles($roles);

            // Clear permission cache
            app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

            DB::commit();

            \Log::info('User updated successfully', [
                'user_id' => $user->id,
                'username' => $user->username,
                'roles' => $user->roles->pluck('name')
            ]);

            return redirect()->route('users.index')
                ->with('success', "User '{$user->name}' updated successfully.");

        } catch (\Exception $e) {
            DB::rollback();

            \Log::error('Failed to update user', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()->withInput()
                ->with('error', 'Failed to update user: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // Prevent deletion of superadmin
        if ($user->hasRole('superadmin')) {
            return back()->with('error', 'Cannot delete superadmin user.');
        }

        // Prevent self deletion
        if ($user->id === auth()->id()) {
            return back()->with('error', 'You cannot delete yourself.');
        }

        DB::beginTransaction();
        try {
            $userName = $user->name;

            // Delete profile photo
            if ($user->profile_photo) {
                Storage::disk('public')->delete($user->profile_photo);
            }

            // Remove all roles before deletion
            $user->syncRoles([]);

            // Delete user
            $user->delete();

            // Clear permission cache
            app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

            DB::commit();

            \Log::info('User deleted successfully', [
                'username' => $userName
            ]);

            return redirect()->route('users.index')
                ->with('success', "User '{$userName}' deleted successfully.");

        } catch (\Exception $e) {
            DB::rollback();

            \Log::error('Failed to delete user', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return back()->with('error', 'Failed to delete user: ' . $e->getMessage());
        }
    }

    /**
     * Show user profile page
     */
    public function showEditProfile(User $user)
    {
        // Only allow users to view their own profile or users with permission
        if (auth()->id() !== $user->id && !auth()->user()->can('user.view')) {
            abort(403, 'You can only view your own profile.');
        }

        $title = 'My Profile';

        return view('admin.users.profile', compact('user', 'title'));
    }

    /**
     * Update user profile
     */
    public function editProfile(Request $request, User $user)
    {
        // Users can only edit their own profile
        if ($user->id !== auth()->id()) {
            abort(403, 'You can only edit your own profile.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('users')->ignore($user->id)
            ],
            'password' => 'nullable|string|min:8|confirmed',
            'profile_photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        try {
            $updateData = [
                'name' => $validated['name'],
                'email' => $validated['email'],
            ];

            // Update password if provided
            if ($request->filled('password')) {
                $updateData['password'] = Hash::make($validated['password']);
            }

            // Handle profile photo upload
            if ($request->hasFile('profile_photo')) {
                // Delete old photo
                if ($user->profile_photo) {
                    Storage::disk('public')->delete($user->profile_photo);
                }

                $file = $request->file('profile_photo');
                $filename = time() . '_' . $file->getClientOriginalName();
                $updateData['profile_photo'] = $file->storeAs('profile_photos', $filename, 'public');
            }

            $user->update($updateData);

            return back()->with('success', 'Profile updated successfully.');

        } catch (\Exception $e) {
            \Log::error('Failed to update profile', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return back()->withInput()
                ->with('error', 'Failed to update profile: ' . $e->getMessage());
        }
    }

    /**
     * Get users by role (for AJAX requests)
     */
    public function getUsersByRole(Request $request)
    {
        $request->validate([
            'role' => 'required|string|exists:roles,name'
        ]);

        $users = User::role($request->role)
            ->select('id', 'name', 'username', 'email')
            ->orderBy('name')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $users
        ]);
    }
}
