<?php

// Middleware: app/Http/Middleware/LogActivity.php
namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Models\ActivityLog;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class LogActivity
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);
        
        // Skip untuk routes tertentu
        $skipRoutes = [
            'activity-logs.index',
            'api/*',
            'assets/*',
            'css/*',
            'js/*',
        ];
        
        $currentRoute = $request->route()->getName();
        
        foreach ($skipRoutes as $skip) {
            if (fnmatch($skip, $currentRoute)) {
                return $response;
            }
        }
        
        // Auto log berdasarkan method dan route
        $this->autoLogActivity($request, $response);
        
        return $response;
    }
    
    private function autoLogActivity(Request $request, Response $response)
    {
        $method = $request->method();
        $route = $request->route()->getName();
        $statusCode = $response->getStatusCode();
        
        // Tentukan status berdasarkan HTTP status code
        $status = match(true) {
            $statusCode >= 200 && $statusCode < 300 => 'Berhasil',
            $statusCode >= 400 && $statusCode < 500 => 'Gagal',
            $statusCode >= 500 => 'Gagal',
            default => 'Berhasil'
        };
        
        // Mapping route ke aktivitas
        $activityMapping = [
            'login' => 'Login',
            'logout' => 'Logout',
            'skpt.store' => 'Create SKPT',
            'skpt.update' => 'Edit SKPT',
            'skpt.destroy' => 'Delete Data',
            'documents.store' => 'Upload Document',
            'documents.approve' => 'Approve Document',
            'users.store' => 'Create User',
        ];
        
        $aktivitas = $activityMapping[$route] ?? ucfirst($method) . ' Request';
        
        // Generate deskripsi otomatis
        $deskripsi = $this->generateDescription($method, $route, $request);
        
        // Tentukan module berdasarkan route
        $module = $this->determineModule($route);
        
        ActivityLog::logActivity($aktivitas, $deskripsi, $module, $status);
    }
    
    private function generateDescription($method, $route, $request)
    {
        $descriptions = [
            'GET' => 'Mengakses halaman ' . str_replace('.', ' ', $route),
            'POST' => 'Menambah data melalui ' . str_replace('.', ' ', $route),
            'PUT' => 'Mengupdate data melalui ' . str_replace('.', ' ', $route),
            'DELETE' => 'Menghapus data melalui ' . str_replace('.', ' ', $route),
        ];
        
        return $descriptions[$method] ?? "Melakukan aksi {$method} pada {$route}";
    }
    
    private function determineModule($route)
    {
        $modules = [
            'login' => 'Authentication',
            'logout' => 'Authentication',
            'skpt' => 'SKPT',
            'documents' => 'Document Management',
            'users' => 'User Management',
            'approval' => 'Approval',
        ];
        
        foreach ($modules as $key => $module) {
            if (strpos($route, $key) !== false) {
                return $module;
            }
        }
        
        return 'System';
    }
}