<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class SKPTUpdateRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $skptId = $this->route('skpt')->id;

        return [
            // Data Pemohon
            'nama_pemohon' => 'required|string|max:255',
            'nik' => [
                'required',
                'string',
                'max:20',
                Rule::unique('skpt_applications', 'nik')->ignore($skptId)
            ],
            'no_ktp' => 'required|string|max:20',
            'tempat_lahir' => 'required|string|max:255',
            'tanggal_lahir' => 'required|date|before:today',
            'email' => 'nullable|email|max:255',
            'telepon' => 'nullable|string|max:15|regex:/^([0-9\s\-\+\(\)]*)$/',
            'pekerjaan' => 'required|string|max:255',
            
            // Alamat Pemohon
            'alamat_rt' => 'required|string|max:3|regex:/^[0-9]{1,3}$/',
            'alamat_rw' => 'required|string|max:3|regex:/^[0-9]{1,3}$/',
            'alamat_kode_pos' => 'required|string|max:5|regex:/^[0-9]{5}$/',
            
            // Detail Tanah
            'letak_tanah' => 'required|string|max:1000',
            'tanah_rt' => 'required|string|max:3|regex:/^[0-9]{1,3}$/',
            'tanah_rw' => 'required|string|max:3|regex:/^[0-9]{1,3}$/',
            'tanah_kode_pos' => 'required|string|max:5|regex:/^[0-9]{5}$/',
            'dusun' => 'required|string|max:255',
            'kelurahan_kampung' => 'required|string|max:255',
            'peruntukan' => 'required|in:Perkebunan,Pertanian,Perumahan,Industri,Lainnya',
            'luas' => 'required|numeric|min:1|max:999999.99',
            'penanda_tanah' => 'required|in:Batas Pagar,Batas Alam,Batas Bangunan,Lainnya',
            'utara' => 'required|in:Tanah Kosong,Rumah,Kebun,Jalan,Sungai,Lainnya',
            'timur' => 'required|in:Tanah Kosong,Rumah,Kebun,Jalan,Sungai,Lainnya',
            'selatan' => 'required|in:Jalan Kampung,Rumah,Kebun,Tanah Kosong,Sungai,Lainnya',
            'barat' => 'required|in:Tanah Hasan,Rumah,Kebun,Tanah Kosong,Sungai,Lainnya',
        ];
    }

    /**
     * Get custom messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'nama_pemohon.required' => 'Nama pemohon harus diisi.',
            'nik.required' => 'NIK harus diisi.',
            'nik.unique' => 'NIK sudah terdaftar dalam sistem.',
            'no_ktp.required' => 'No. KTP harus diisi.',
            'tempat_lahir.required' => 'Tempat lahir harus diisi.',
            'tanggal_lahir.required' => 'Tanggal lahir harus diisi.',
            'tanggal_lahir.before' => 'Tanggal lahir harus sebelum hari ini.',
            'email.email' => 'Format email tidak valid.',
            'telepon.regex' => 'Format nomor telepon tidak valid.',
            'pekerjaan.required' => 'Pekerjaan harus diisi.',
            
            'alamat_rt.required' => 'RT alamat pemohon harus diisi.',
            'alamat_rt.regex' => 'RT harus berupa angka.',
            'alamat_rw.required' => 'RW alamat pemohon harus diisi.',
            'alamat_rw.regex' => 'RW harus berupa angka.',
            'alamat_kode_pos.required' => 'Kode pos alamat pemohon harus diisi.',
            'alamat_kode_pos.regex' => 'Kode pos harus 5 digit angka.',
            
            'letak_tanah.required' => 'Letak tanah harus diisi.',
            'tanah_rt.required' => 'RT lokasi tanah harus diisi.',
            'tanah_rt.regex' => 'RT harus berupa angka.',
            'tanah_rw.required' => 'RW lokasi tanah harus diisi.',
            'tanah_rw.regex' => 'RW harus berupa angka.',
            'tanah_kode_pos.required' => 'Kode pos lokasi tanah harus diisi.',
            'tanah_kode_pos.regex' => 'Kode pos harus 5 digit angka.',
            'dusun.required' => 'Dusun harus diisi.',
            'kelurahan_kampung.required' => 'Kelurahan/Kampung harus diisi.',
            'peruntukan.required' => 'Peruntukan tanah harus dipilih.',
            'luas.required' => 'Luas tanah harus diisi.',
            'luas.min' => 'Luas tanah minimal 1 m².',
            'luas.max' => 'Luas tanah maksimal 999,999.99 m².',
            'penanda_tanah.required' => 'Penanda tanah harus dipilih.',
            'utara.required' => 'Batas utara harus dipilih.',
            'timur.required' => 'Batas timur harus dipilih.',
            'selatan.required' => 'Batas selatan harus dipilih.',
            'barat.required' => 'Batas barat harus dipilih.',
        ];
    }
}