<?php

// ==========================================
// 1. SpSiptRequest.php - Main Form Request
// ==========================================

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use App\Models\SpSipt;

class SpSiptRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        // Add your authorization logic here
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            // Data Pemohon (Required)
            'pemohon_nama' => 'required|string|max:255',
            'pemohon_nik' => 'required|digits:16|numeric',
            'pemohon_tempat_lahir' => 'nullable|string|max:100',
            'pemohon_tanggal_lahir' => 'nullable|date|before:today',
            'pemohon_alamat' => 'required|string|max:500',
            'pemohon_rt' => 'nullable|string|max:5',
            'pemohon_rw' => 'nullable|string|max:5',
            'pemohon_kelurahan' => 'nullable|string|max:100',
            'pemohon_kecamatan' => 'nullable|string|max:100',
            'pemohon_kota' => 'nullable|string|max:100',
            'pemohon_provinsi' => 'nullable|string|max:100',
            'pemohon_kode_pos' => 'nullable|string|max:10',
            'pemohon_telepon' => 'nullable|string|max:20',
            'pemohon_hp' => 'nullable|string|max:20|regex:/^([0-9\s\-\+\(\)]*)$/',
            'pemohon_email' => 'nullable|email|max:100',
            
            // Data Terlapor (Required)
            'terlapor_nama' => 'required|string|max:255',
            'terlapor_nik' => 'nullable|digits:16|numeric',
            'terlapor_tempat_lahir' => 'nullable|string|max:100',
            'terlapor_tanggal_lahir' => 'nullable|date|before:today',
            'terlapor_alamat' => 'nullable|string|max:500',
            'terlapor_rt' => 'nullable|string|max:5',
            'terlapor_rw' => 'nullable|string|max:5',
            'terlapor_kelurahan' => 'nullable|string|max:100',
            'terlapor_kecamatan' => 'nullable|string|max:100',
            'terlapor_kota' => 'nullable|string|max:100',
            'terlapor_provinsi' => 'nullable|string|max:100',
            'terlapor_kode_pos' => 'nullable|string|max:10',
            'terlapor_telepon' => 'nullable|string|max:20',
            'terlapor_hp' => 'nullable|string|max:20|regex:/^([0-9\s\-\+\(\)]*)$/',
            
            // Data Tanah
            'status_tanah' => 'nullable|in:hak_milik,hak_guna_bangunan,hak_guna_usaha,hak_pakai,tanah_negara,lainnya',
            'status_tanah_lainnya' => 'required_if:status_tanah,lainnya|nullable|string|max:100',
            'nomor_sertifikat' => 'nullable|string|max:100',
            'tanggal_sertifikat' => 'nullable|date|before_or_equal:today',
            'nama_pemilik_sertifikat' => 'nullable|string|max:255',
            
            // Lokasi Tanah
            'lokasi_jalan' => 'nullable|string|max:255',
            'lokasi_rt' => 'nullable|string|max:5',
            'lokasi_rw' => 'nullable|string|max:5',
            'lokasi_kelurahan' => 'nullable|string|max:100',
            'lokasi_kecamatan' => 'nullable|string|max:100',
            'lokasi_kota' => 'nullable|string|max:100',
            'lokasi_provinsi' => 'nullable|string|max:100',
            
            // Ukuran Tanah
            'luas_tanah' => 'nullable|numeric|min:0|max:9999999.99',
            'batas_utara' => 'nullable|string|max:255',
            'batas_selatan' => 'nullable|string|max:255',
            'batas_timur' => 'nullable|string|max:255',
            'batas_barat' => 'nullable|string|max:255',
            
            // Keterangan Pengaduan
            'kronologi' => 'required|string|min:10|max:5000',
            'tuntutan' => 'nullable|string|max:2000',
            'dasar_hukum' => 'nullable|string|max:2000',
            
            // Persetujuan
            'pernyataan_kebenaran' => 'boolean',
            'kesediaan_mediasi' => 'boolean',
            
            // Koordinat (for map)
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'polygon_coordinates' => 'nullable|json',
            
            // Files (handled separately)
            'ttd_pemohon' => 'nullable|image|max:2048', // Max 2MB
            'ttd_terlapor' => 'nullable|image|max:2048',
            
            // Document uploads
            'ktp_pemohon' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120', // Max 5MB
            'ktp_terlapor' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'sertifikat_tanah' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:10240', // Max 10MB
            'surat_ukur' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:10240',
            'imb' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:10240',
            'pbb' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'akta_jual_beli' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:10240',
            'surat_waris' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:10240',
            'surat_hibah' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:10240',
            'putusan_pengadilan' => 'nullable|file|mimes:pdf|max:20480', // Max 20MB
            'berita_acara' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:10240',
            'foto_lokasi' => 'nullable|array',
            'foto_lokasi.*' => 'image|max:5120', // Max 5MB per image
            
            // Multiple documents
            'dokumen' => 'nullable|array',
            'dokumen.*' => 'file|max:10240',
            
            // Submit action
            'submit_action' => 'nullable|in:draft,submit'
        ];

        // If updating, make some fields not required
        if ($this->isMethod('PUT') || $this->isMethod('PATCH')) {
            $rules['pemohon_nama'] = 'sometimes|required|string|max:255';
            $rules['pemohon_nik'] = 'sometimes|required|digits:16|numeric';
            $rules['pemohon_alamat'] = 'sometimes|required|string|max:500';
            $rules['terlapor_nama'] = 'sometimes|required|string|max:255';
            $rules['kronologi'] = 'sometimes|required|string|min:10|max:5000';
        }

        return $rules;
    }

    /**
     * Get custom error messages
     */
    public function messages(): array
    {
        return [
            'pemohon_nama.required' => 'Nama pemohon wajib diisi.',
            'pemohon_nik.required' => 'NIK pemohon wajib diisi.',
            'pemohon_nik.digits' => 'NIK harus terdiri dari 16 digit.',
            'pemohon_alamat.required' => 'Alamat pemohon wajib diisi.',
            'terlapor_nama.required' => 'Nama pihak yang diadukan wajib diisi.',
            'kronologi.required' => 'Kronologi permasalahan wajib diisi.',
            'kronologi.min' => 'Kronologi minimal 10 karakter.',
            'status_tanah_lainnya.required_if' => 'Keterangan status tanah lainnya wajib diisi.',
            'pemohon_hp.regex' => 'Format nomor HP tidak valid.',
            'terlapor_hp.regex' => 'Format nomor HP tidak valid.',
            'latitude.between' => 'Latitude harus antara -90 dan 90.',
            'longitude.between' => 'Longitude harus antara -180 dan 180.',
            'ttd_pemohon.image' => 'Tanda tangan pemohon harus berupa gambar.',
            'ttd_pemohon.max' => 'Ukuran tanda tangan maksimal 2MB.',
            'ktp_pemohon.mimes' => 'KTP pemohon harus berformat JPG, PNG, atau PDF.',
            'foto_lokasi.*.image' => 'Foto lokasi harus berupa gambar.',
            'foto_lokasi.*.max' => 'Ukuran foto lokasi maksimal 5MB per file.',
        ];
    }

    /**
     * Get custom attribute names
     */
    public function attributes(): array
    {
        return [
            'pemohon_nama' => 'nama pemohon',
            'pemohon_nik' => 'NIK pemohon',
            'pemohon_tempat_lahir' => 'tempat lahir pemohon',
            'pemohon_tanggal_lahir' => 'tanggal lahir pemohon',
            'pemohon_alamat' => 'alamat pemohon',
            'pemohon_hp' => 'nomor HP pemohon',
            'terlapor_nama' => 'nama terlapor',
            'terlapor_nik' => 'NIK terlapor',
            'luas_tanah' => 'luas tanah',
            'ttd_pemohon' => 'tanda tangan pemohon',
            'ttd_terlapor' => 'tanda tangan terlapor',
        ];
    }

    /**
     * Prepare data for validation
     */
    protected function prepareForValidation(): void
    {
        // Convert checkbox values
        $this->merge([
            'pernyataan_kebenaran' => $this->boolean('pernyataan_kebenaran'),
            'kesediaan_mediasi' => $this->boolean('kesediaan_mediasi'),
        ]);

        // Clean NIK (remove spaces or special chars)
        if ($this->has('pemohon_nik')) {
            $this->merge([
                'pemohon_nik' => preg_replace('/[^0-9]/', '', $this->pemohon_nik)
            ]);
        }

        if ($this->has('terlapor_nik')) {
            $this->merge([
                'terlapor_nik' => preg_replace('/[^0-9]/', '', $this->terlapor_nik)
            ]);
        }

        // Clean phone numbers
        if ($this->has('pemohon_hp')) {
            $this->merge([
                'pemohon_hp' => preg_replace('/[^0-9+]/', '', $this->pemohon_hp)
            ]);
        }

        if ($this->has('terlapor_hp')) {
            $this->merge([
                'terlapor_hp' => preg_replace('/[^0-9+]/', '', $this->terlapor_hp)
            ]);
        }
    }
}

