<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;

class ActivityLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'waktu',
        'user_name',
        'user_role',
        'aktivitas',
        'deskripsi',
        'module',
        'status',
        'ip_address',
        'user_id',
        'browser',
        'device',
        'additional_data'
    ];

    protected $casts = [
        'waktu' => 'datetime',
        'additional_data' => 'array',
    ];

    // Relationship dengan User
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    // Scope untuk filter berdasarkan status
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    // Scope untuk filter berdasarkan module
    public function scopeByModule($query, $module)
    {
        return $query->where('module', $module);
    }

    // Scope untuk filter berdasarkan tanggal
    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('waktu', [$startDate, $endDate]);
    }

    // Method untuk log aktivitas
    public static function logActivity($aktivitas, $deskripsi, $module, $status = 'Berhasil', $additionalData = null)
    {
        $user = Auth::user();
        
        return self::create([
            'waktu' => now(),
            'user_name' => $user ? $user->name : 'Guest User',
            'user_role' => $user ? '' ?? 'Guest' : 'Guest',
            'aktivitas' => $aktivitas,
            'deskripsi' => $deskripsi,
            'module' => $module,
            'status' => $status,
            'ip_address' => Request::ip(),
            'user_id' => $user ? $user->id : null,
            'browser' => Request::header('User-Agent'),
            'device' => self::getDeviceInfo(),
            'additional_data' => $additionalData
        ]);
    }

    // Helper untuk mendapatkan info device
    private static function getDeviceInfo()
    {
        $userAgent = Request::header('User-Agent');
        
        if (strpos($userAgent, 'Mobile') !== false) {
            return 'Mobile';
        } elseif (strpos($userAgent, 'Tablet') !== false) {
            return 'Tablet';
        } else {
            return 'Desktop';
        }
    }

    // Method untuk mendapatkan icon berdasarkan aktivitas
    public function getActivityIcon()
    {
        $icons = [
            'Login' => '🔐',
            'Logout' => '🚪',
            'Create SKPT' => '📝',
            'Edit SKPT' => '✏️',
            'Delete Data' => '🗑️',
            'Approve Document' => '✅',
            'Upload Document' => '📤',
            'Print Document' => '🖨️',
            'Create User' => '👤',
            'Failed Login' => '❌',
        ];

        return $icons[$this->aktivitas] ?? '📋';
    }

    // Method untuk mendapatkan warna status
    public function getStatusColor()
    {
        return match($this->status) {
            'Berhasil' => 'success',
            'Gagal' => 'danger',
            'Peringatan' => 'warning',
            default => 'secondary'
        };
    }
}

// Helper Class untuk memudahkan logging
class ActivityLogger
{
    public static function login($success = true)
    {
        $status = $success ? 'Berhasil' : 'Gagal';
        $aktivitas = $success ? 'Login' : 'Failed Login';
        $deskripsi = $success ? 'User berhasil login ke sistem' : 'Percobaan login gagal - password salah';
        
        ActivityLog::logActivity($aktivitas, $deskripsi, 'Authentication', $status);
    }

    public static function logout()
    {
        ActivityLog::logActivity('Logout', 'User berhasil logout dari sistem', 'Authentication');
    }

    public static function createSKPT($nik)
    {
        ActivityLog::logActivity(
            'Create SKPT',
            "Membuat permohonan SKPT baru dengan NIK {$nik}",
            'SKPT',
            'Berhasil',
            ['nik' => $nik]
        );
    }

    public static function editSKPT($nik)
    {
        ActivityLog::logActivity(
            'Edit SKPT',
            "Mengedit data SKPT dengan NIK {$nik}",
            'SKPT',
            'Berhasil',
            ['nik' => $nik]
        );
    }

    public static function deleteData($dataType, $identifier)
    {
        ActivityLog::logActivity(
            'Delete Data',
            "Menghapus data {$dataType} dengan ID {$identifier}",
            'Master Data',
            'Peringatan',
            ['data_type' => $dataType, 'identifier' => $identifier]
        );
    }

    public static function approveDocument($documentNumber)
    {
        ActivityLog::logActivity(
            'Approve Document',
            "Menyetujui persetujuan SKPT dengan nomor {$documentNumber}",
            'Approval',
            'Berhasil',
            ['document_number' => $documentNumber]
        );
    }

    public static function uploadDocument($documentType)
    {
        ActivityLog::logActivity(
            'Upload Document',
            "Mengupload dokumen pendukung untuk permohonan {$documentType}",
            $documentType,
            'Berhasil'
        );
    }

    public static function printDocument($documentNumber)
    {
        ActivityLog::logActivity(
            'Print Document',
            "Mencetak dokumen SKPT dengan nomor {$documentNumber}",
            'SKPT',
            'Berhasil',
            ['document_number' => $documentNumber]
        );
    }

    public static function createUser($userRole)
    {
        ActivityLog::logActivity(
            'Create User',
            "Membuat user baru dengan role {$userRole}",
            'User Management',
            'Berhasil',
            ['role' => $userRole]
        );
    }
}