<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Wildside\Userstamps\Userstamps;
use Illuminate\Database\Eloquent\SoftDeletes;

class MsService extends Model
{
    use HasFactory, Userstamps, SoftDeletes;
    
    protected $table = 'ms_service';
    protected $guarded = [];

    protected $casts = [
        'harga' => 'decimal:2',
        'ppn' => 'decimal:2',
    ];

    // Relationships
    public function category()
    {
        return $this->belongsTo(MsServiceCategory::class, 'service_category_id');
    }

    public function bookingServices()
    {
        return $this->hasMany(BookingService::class, 'service_id');
    }

    // Accessors
    public function getFormattedPriceAttribute()
    {
        return 'Rp ' . number_format($this->harga, 0, ',', '.');
    }

    public function getFormattedPpnAttribute()
    {
        return number_format($this->ppn, 1) . '%';
    }

    public function getPriceWithPpnAttribute()
    {
        $ppnAmount = ($this->harga * $this->ppn) / 100;
        return $this->harga + $ppnAmount;
    }

    public function getFormattedPriceWithPpnAttribute()
    {
        return 'Rp ' . number_format($this->price_with_ppn, 0, ',', '.');
    }

    public function getPpnAmountAttribute()
    {
        return ($this->harga * $this->ppn) / 100;
    }

    public function getFormattedPpnAmountAttribute()
    {
        return 'Rp ' . number_format($this->ppn_amount, 0, ',', '.');
    }

    // Scopes
    public function scopeByCategory($query, $categoryId)
    {
        return $query->where('service_category_id', $categoryId);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('name', 'like', '%' . $search . '%')
              ->orWhere('satuan', 'like', '%' . $search . '%')
              ->orWhereHas('category', function($subQ) use ($search) {
                  $subQ->where('name', 'like', '%' . $search . '%');
              });
        });
    }

    public function scopePriceRange($query, $minPrice = null, $maxPrice = null)
    {
        if ($minPrice !== null) {
            $query->where('harga', '>=', $minPrice);
        }
        
        if ($maxPrice !== null) {
            $query->where('harga', '<=', $maxPrice);
        }
        
        return $query;
    }

    // Methods
    public function calculateTotal($quantity)
    {
        $subtotal = $this->harga * $quantity;
        $ppnAmount = ($subtotal * $this->ppn) / 100;
        
        return [
            'quantity' => $quantity,
            'unit_price' => $this->harga,
            'subtotal' => $subtotal,
            'ppn_rate' => $this->ppn,
            'ppn_amount' => $ppnAmount,
            'total' => $subtotal + $ppnAmount
        ];
    }
}