<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class SKPTApplication extends Model
{
    use HasFactory;

    protected $table = 'skpt_applications';
    protected $guarded = [''];
    protected $dates = [
        'tgl_penerimaan',
        'tgl_permohonan',
        'tanggal_lahir',
        'created_at',
        'updated_at'
    ];

    protected $casts = [
        'tgl_penerimaan' => 'date',
        'tgl_permohonan' => 'date',
        'tanggal_lahir' => 'date',
        'luas' => 'decimal:2'
    ];

    public function anak()
    {
        return $this->hasMany(SKPTAnak::class, 'skpt_id');
    }

    public function dataPeninjauan()
    {
        return $this->hasOne(Peninjauan::class, 'skpt_id', 'id');
    }

    // Relasi dengan dokumen
    public function documents()
    {
        return $this->hasMany(SKPTDocument::class, 'skpt_application_id');
    }

    public function additionalDocuments()
    {
        return $this->hasMany(SKPTAdditionalDocument::class,'skpt_application_id');
    }

    // Accessor untuk alamat lengkap pemohon
    public function getAlamatLengkapPemohonAttribute()
    {
        return "RT {$this->alamat_rt} RW {$this->alamat_rw} Kode Pos {$this->alamat_kode_pos}";
    }

    // Accessor untuk alamat lengkap tanah
    public function getAlamatLengkapTanahAttribute()
    {
        return "RT {$this->tanah_rt} RW {$this->tanah_rw} Kode Pos {$this->tanah_kode_pos}";
    }

    // Accessor untuk format luas tanah
    public function getLuasFormatAttribute()
    {
        return number_format($this->luas, 0, ',', '.') . ' m²';
    }

    // Accessor untuk umur
    public function getUmurAttribute()
    {
        return Carbon::parse($this->tanggal_lahir)->age;
    }

    // Scope untuk filter berdasarkan status
    public function scopeByStatus($query, $status)
    {
        return $query->where('status_permohonan', $status);
    }

    public function scopeByKelengkapanDokumen($query, $status)
    {
        return $query->where('status_kelengkapan_dokumen', $status);
    }

    // Scope untuk pencarian
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('nama_pemohon', 'LIKE', "%{$search}%")
              ->orWhere('nik', 'LIKE', "%{$search}%")
              ->orWhere('no_ktp', 'LIKE', "%{$search}%");
        });
    }

    // Method untuk check kelengkapan dokumen
    public function isDocumentComplete()
    {
        $requiredDocs = $this->documents()->where('is_required', true)->count();
        $uploadedRequiredDocs = $this->documents()
            ->where('is_required', true)
            ->where('is_uploaded', true)
            ->count();

        return $requiredDocs > 0 && $requiredDocs === $uploadedRequiredDocs;
    }

   public function isAllRequiredDocumentsVerified()
    {
        // Ambil semua dokumen wajib (tanpa memfilter is_uploaded terlebih dahulu)
        $requiredDocs = $this->documents()->where('is_required', true);

        // Hitung total dokumen wajib yang sudah diunggah
        $uploadedRequiredDocs = $requiredDocs->where('is_uploaded', true)->count();

        // Hitung dokumen wajib yang sudah diunggah dan diverifikasi
        $verifiedUploadedDocs = $requiredDocs->where('is_uploaded', true)->where('verification_status', 'SUDAH')->count();

        // Kembalikan true hanya jika ada dokumen wajib yang diunggah dan semua yang diunggah sudah diverifikasi
        return $uploadedRequiredDocs > 0 && $uploadedRequiredDocs === $verifiedUploadedDocs;
    }

    // Method untuk mendapatkan statistik dokumen
    public function getDocumentStatistics()
    {
        $required = $this->documents()->required();
        $optional = $this->documents()->optional();
        $additional = $this->additionalDocuments();

        return [
            'required' => [
                'total' => $required->count(),
                'uploaded' => $required->where('is_uploaded', true)->count(),
                'verified' => $required->verified()->count(),
                'pending' => $required->pending()->count(),
                'rejected' => $required->rejected()->count(),
            ],
            'optional' => [
                'total' => $optional->count(),
                'uploaded' => $optional->where('is_uploaded', true)->count(),
                'verified' => $optional->verified()->count(),
                'pending' => $optional->pending()->count(),
                'rejected' => $optional->rejected()->count(),
            ],
            'additional' => [
                'total' => $additional->count(),
                'verified' => $additional->verified()->count(),
                'pending' => $additional->pending()->count(),
                'rejected' => $additional->rejected()->count(),
            ]
        ];
    }

    // Method untuk update status kelengkapan dokumen
    public function updateDocumentStatus()
    {
        // $this->status_kelengkapan_dokumen = $this->isDocumentComplete() ? 'LENGKAP' : 'BELUM LENGKAP';
        if ($this->isAllRequiredDocumentsVerified()) {
            $this->status_kelengkapan_dokumen = 'LENGKAP';
        } else {
            $this->status_kelengkapan_dokumen = 'BELUM LENGKAP';
        }
        $this->save();
    }

    // Method untuk auto update status permohonan berdasarkan verifikasi
    public function updateApplicationStatus()
    {
        if ($this->isAllRequiredDocumentsVerified()) {
            $this->status_permohonan = 'SELESAI';
        } else {
            $this->status_permohonan = 'BELUM SELESAI';
        }
        $this->save();
    }

    // Boot method untuk auto-generate beberapa field
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            // Auto-generate no_registrasi jika belum ada
            if (!$model->no_registrasi) {
                $model->no_registrasi = $model->generateNoRegistrasi();
            }

            if (!$model->tgl_penerimaan) {
                $model->tgl_penerimaan = now();
            }
            if (!$model->tgl_permohonan) {
                $model->tgl_permohonan = now();
            }
        });
    }

    // Method untuk generate nomor registrasi
    public function generateNoRegistrasi()
    {
        $year = date('Y');
        $month = date('m');

        // Cari nomor urut terakhir untuk bulan dan tahun ini
        $lastApplication = self::whereYear('created_at', $year)
                              ->whereMonth('created_at', $month)
                              ->orderBy('id', 'desc')
                              ->first();

        $sequence = 1;
        if ($lastApplication && $lastApplication->no_registrasi) {
            // Extract nomor urut dari format XXX-YYY-ZZZ
            $parts = explode('-', $lastApplication->no_registrasi);
            if (count($parts) === 3) {
                $sequence = (int)$parts[2] + 1;
            }
        }

        // Format: XXX-YYY-ZZZ
        // XXX = 3 digit urut sequence
        // YYY = 3 digit bulan+tahun (contoh: 025 untuk Feb 2025)
        // ZZZ = 3 digit nomor urut dalam bulan
        $yearMonth = str_pad($month . substr($year, -2), 3, '0', STR_PAD_LEFT);

        return sprintf('%03d-%s-%03d',
            rand(1, 999), // Prefix acak atau bisa diganti dengan kode wilayah
            $yearMonth,
            $sequence
        );
    }

    // Method untuk generate nomor registrasi custom
    public static function generateCustomNoRegistrasi($prefix = null, $sequence = null)
    {
        $year = date('Y');
        $month = date('m');

        if (!$prefix) {
            $prefix = str_pad(rand(1, 999), 3, '0', STR_PAD_LEFT);
        }

        if (!$sequence) {
            $lastApplication = self::whereYear('created_at', $year)
                                  ->whereMonth('created_at', $month)
                                  ->orderBy('id', 'desc')
                                  ->first();

            $sequence = 1;
            if ($lastApplication && $lastApplication->no_registrasi) {
                $parts = explode('-', $lastApplication->no_registrasi);
                if (count($parts) === 3) {
                    $sequence = (int)$parts[2] + 1;
                }
            }
        }

        $yearMonth = str_pad($month . substr($year, -2), 3, '0', STR_PAD_LEFT);

        return sprintf('%03d-%s-%03d', $prefix, $yearMonth, $sequence);
    }
}
