<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class SpSiptJadwal extends Model
{
    use HasFactory;

    protected $table = 'sp_sipt_jadwal';

    protected $fillable = [
        'sp_sipt_id',
        'tanggal_waktu',
        'tempat',
        'jenis_kegiatan',
        'agenda',
        'hasil',
        'status',
        'created_by'
    ];

    protected $casts = [
        'tanggal_waktu' => 'datetime'
    ];

    /**
     * Jenis kegiatan constants
     */
    const KEGIATAN_MEDIASI = 'mediasi';
    const KEGIATAN_SURVEY = 'survey';
    const KEGIATAN_PERTEMUAN = 'pertemuan';
    const KEGIATAN_SIDANG = 'sidang';

    /**
     * Status constants
     */
    const STATUS_SCHEDULED = 'scheduled';
    const STATUS_ONGOING = 'ongoing';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';

    /**
     * Relationships
     */
    public function spSipt()
    {
        return $this->belongsTo(SpSipt::class, 'sp_sipt_id');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Scopes
     */
    public function scopeUpcoming($query)
    {
        return $query->where('tanggal_waktu', '>=', now())
                     ->where('status', self::STATUS_SCHEDULED);
    }

    public function scopePast($query)
    {
        return $query->where('tanggal_waktu', '<', now());
    }

    public function scopeToday($query)
    {
        return $query->whereDate('tanggal_waktu', today());
    }

    public function scopeThisWeek($query)
    {
        return $query->whereBetween('tanggal_waktu', [
            now()->startOfWeek(),
            now()->endOfWeek()
        ]);
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Accessors
     */
    public function getJenisKegiatanLabelAttribute()
    {
        return match($this->jenis_kegiatan) {
            self::KEGIATAN_MEDIASI => 'Mediasi',
            self::KEGIATAN_SURVEY => 'Survey Lapangan',
            self::KEGIATAN_PERTEMUAN => 'Pertemuan',
            self::KEGIATAN_SIDANG => 'Sidang',
            default => 'Unknown'
        };
    }

    public function getStatusLabelAttribute()
    {
        return match($this->status) {
            self::STATUS_SCHEDULED => 'Dijadwalkan',
            self::STATUS_ONGOING => 'Sedang Berlangsung',
            self::STATUS_COMPLETED => 'Selesai',
            self::STATUS_CANCELLED => 'Dibatalkan',
            default => 'Unknown'
        };
    }

    public function getStatusBadgeClassAttribute()
    {
        return match($this->status) {
            self::STATUS_SCHEDULED => 'badge-info',
            self::STATUS_ONGOING => 'badge-warning',
            self::STATUS_COMPLETED => 'badge-success',
            self::STATUS_CANCELLED => 'badge-danger',
            default => 'badge-light'
        };
    }

    public function getIsUpcomingAttribute()
    {
        return $this->tanggal_waktu->isFuture() && $this->status == self::STATUS_SCHEDULED;
    }

    public function getIsPastAttribute()
    {
        return $this->tanggal_waktu->isPast();
    }

    public function getIsTodayAttribute()
    {
        return $this->tanggal_waktu->isToday();
    }

    public function getFormattedDateAttribute()
    {
        return $this->tanggal_waktu->format('d M Y');
    }

    public function getFormattedTimeAttribute()
    {
        return $this->tanggal_waktu->format('H:i');
    }

    public function getFormattedDateTimeAttribute()
    {
        return $this->tanggal_waktu->format('d M Y, H:i');
    }

    /**
     * Methods
     */
    public function canBeCancelled()
    {
        return $this->status == self::STATUS_SCHEDULED && $this->tanggal_waktu->isFuture();
    }

    public function cancel($reason = null)
    {
        if (!$this->canBeCancelled()) {
            return false;
        }

        $this->status = self::STATUS_CANCELLED;
        
        if ($reason) {
            $this->hasil = "Dibatalkan: {$reason}";
        }
        
        $this->save();
        
        return true;
    }

    public function markAsCompleted($hasil = null)
    {
        $this->status = self::STATUS_COMPLETED;
        
        if ($hasil) {
            $this->hasil = $hasil;
        }
        
        $this->save();
        
        return true;
    }

    public function markAsOngoing()
    {
        $this->status = self::STATUS_ONGOING;
        $this->save();
        
        return true;
    }

    public function sendReminder()
    {
        // Implement notification logic here
        // Could send email/SMS to relevant parties
    }
}
