<style>
/* ================================
   Enhanced Modern App Styles
   Clean, Elegant, & Professional
   ================================ */

:root {
    /* Color Palette - Modern & Sophisticated */
    --primary: #3b82f6;
    --primary-hover: #2563eb;
    --primary-light: #dbeafe;
    --secondary: #64748b;
    --secondary-light: #f1f5f9;
    --success: #10b981;
    --success-light: #d1fae5;
    --warning: #f59e0b;
    --warning-light: #fef3c7;
    --danger: #ef4444;
    --danger-light: #fee2e2;
    --info: #06b6d4;
    --info-light: #cffafe;

    /* Neutrals */
    --white: #ffffff;
    --gray-50: #f8fafc;
    --gray-100: #f1f5f9;
    --gray-200: #e2e8f0;
    --gray-300: #cbd5e1;
    --gray-400: #94a3b8;
    --gray-500: #64748b;
    --gray-600: #475569;
    --gray-700: #334155;
    --gray-800: #1e293b;
    --gray-900: #0f172a;

    /* Typography */
    --font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
    --font-size-xs: 0.75rem;
    --font-size-sm: 0.875rem;
    --font-size-base: 1rem;
    --font-size-lg: 1.125rem;
    --font-size-xl: 1.25rem;
    --font-size-2xl: 1.5rem;
    --font-size-3xl: 1.875rem;

    /* Spacing */
    --spacing-1: 0.25rem;
    --spacing-2: 0.5rem;
    --spacing-3: 0.75rem;
    --spacing-4: 1rem;
    --spacing-5: 1.25rem;
    --spacing-6: 1.5rem;
    --spacing-8: 2rem;
    --spacing-10: 2.5rem;
    --spacing-12: 3rem;

    /* Border Radius */
    --radius-sm: 0.375rem;
    --radius: 0.5rem;
    --radius-md: 0.75rem;
    --radius-lg: 1rem;
    --radius-xl: 1.5rem;
    --radius-full: 9999px;

    /* Shadows */
    --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
    --shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
    --shadow-xl: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
}

/* ================================
   Base Styles
   ================================ */

* {
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    font-size: var(--font-size-base);
    line-height: 1.6;
    color: var(--gray-800);
    background-color: var(--gray-50);
    margin: 0;
    padding: 0;
    -webkit-font-smoothing: antialiased;
    -moz-osx-font-smoothing: grayscale;
}

/* ================================
   Navigation Styles
   ================================ */

.top-navbar {
    background: var(--white);
    border-bottom: 1px solid var(--gray-200);
    padding: var(--spacing-4) 0;
    box-shadow: var(--shadow-sm);
    backdrop-filter: blur(10px);
    position: sticky;
    top: 0;
    z-index: 100;
}

.navbar-brand {
    display: flex;
    align-items: center;
    font-weight: 700;
    font-size: var(--font-size-lg);
    color: var(--gray-900) !important;
    text-decoration: none;
    transition: all 0.2s ease;
}

.navbar-brand:hover {
    color: var(--primary) !important;
}

.brand-logo {
    width: 40px;
    height: 40px;
    background: linear-gradient(135deg, var(--primary), var(--primary-hover));
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: var(--spacing-3);
    box-shadow: var(--shadow);
}

.brand-logo i {
    color: var(--white);
    font-size: 18px;
}

.nav-menu {
    display: flex;
    align-items: center;
    gap: var(--spacing-2);
}

.nav-link {
    color: var(--gray-700) !important;
    font-weight: 500;
    font-size: var(--font-size-sm);
    padding: var(--spacing-3) var(--spacing-4) !important;
    border-radius: var(--radius);
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    gap: var(--spacing-2);
    position: relative;
    text-decoration: none;
}

.nav-link:hover {
    background: var(--primary-light);
    color: var(--primary) !important;
    transform: translateY(-1px);
}

.nav-link.active {
    background: var(--primary);
    color: var(--white) !important;
    box-shadow: var(--shadow);
}

.nav-link.active::after {
    content: '';
    position: absolute;
    bottom: -1px;
    left: 50%;
    transform: translateX(-50%);
    width: 0;
    height: 0;
    border-left: 6px solid transparent;
    border-right: 6px solid transparent;
    border-bottom: 6px solid var(--primary);
}

.user-menu {
    display: flex;
    align-items: center;
    gap: var(--spacing-2);
    background: linear-gradient(135deg, var(--primary), var(--primary-hover));
    color: var(--white);
    padding: var(--spacing-3) var(--spacing-4);
    border-radius: var(--radius-md);
    font-weight: 500;
    font-size: var(--font-size-sm);
    cursor: pointer;
    border: none;
    transition: all 0.2s ease;
    box-shadow: var(--shadow);
}

.user-menu:hover {
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: var(--white);
}

/* ================================
   Cards & Containers
   ================================ */

.card {
    background: var(--white);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow);
    border: 1px solid var(--gray-200);
    overflow: hidden;
    transition: all 0.3s ease;
    margin-bottom: var(--spacing-6);
}

.card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
}

.card-header {
    padding: var(--spacing-5) var(--spacing-6);
    background: linear-gradient(135deg, var(--gray-50), var(--white));
    border-bottom: 1px solid var(--gray-200);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.card-header h1,
.card-header h2,
.card-header h3,
.card-header h4,
.card-header h5,
.card-header h6 {
    margin: 0;
    font-weight: 600;
    color: var(--gray-900);
    display: flex;
    align-items: center;
    gap: var(--spacing-2);
}

.card-body {
    padding: var(--spacing-6);
}

.card-footer {
    padding: var(--spacing-4) var(--spacing-6);
    background: var(--gray-50);
    border-top: 1px solid var(--gray-200);
}

/* Special card variants */
.card-primary {
    border-left: 4px solid var(--primary);
}

.card-success {
    border-left: 4px solid var(--success);
}

.card-warning {
    border-left: 4px solid var(--warning);
}

.card-danger {
    border-left: 4px solid var(--danger);
}

.card-info {
    border-left: 4px solid var(--info);
}

/* ================================
   Buttons
   ================================ */

.btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: var(--spacing-2);
    padding: var(--spacing-3) var(--spacing-4);
    border: none;
    border-radius: var(--radius);
    font-size: var(--font-size-sm);
    font-weight: 500;
    text-decoration: none;
    cursor: pointer;
    transition: all 0.2s ease;
    position: relative;
    overflow: hidden;
    white-space: nowrap;
}

.btn:focus {
    outline: none;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none !important;
}

/* Button Sizes */
.btn-xs {
    padding: var(--spacing-1) var(--spacing-2);
    font-size: var(--font-size-xs);
}

.btn-sm {
    padding: var(--spacing-2) var(--spacing-3);
    font-size: var(--font-size-sm);
}

.btn-lg {
    padding: var(--spacing-4) var(--spacing-6);
    font-size: var(--font-size-lg);
}

.btn-xl {
    padding: var(--spacing-5) var(--spacing-8);
    font-size: var(--font-size-xl);
}

/* Button Variants */
.btn-primary {
    background: linear-gradient(135deg, var(--primary), var(--primary-hover));
    color: var(--white);
    box-shadow: var(--shadow);
}

.btn-primary:hover {
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: var(--white);
}

.btn-secondary {
    background: var(--gray-100);
    color: var(--gray-700);
    border: 1px solid var(--gray-300);
}

.btn-secondary:hover {
    background: var(--gray-200);
    border-color: var(--gray-400);
    color: var(--gray-800);
    transform: translateY(-1px);
}

.btn-success {
    background: linear-gradient(135deg, var(--success), #059669);
    color: var(--white);
    box-shadow: var(--shadow);
}

.btn-success:hover {
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: var(--white);
}

.btn-warning {
    background: linear-gradient(135deg, var(--warning), #d97706);
    color: var(--white);
    box-shadow: var(--shadow);
}

.btn-warning:hover {
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: var(--white);
}

.btn-danger {
    background: linear-gradient(135deg, var(--danger), #dc2626);
    color: var(--white);
    box-shadow: var(--shadow);
}

.btn-danger:hover {
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: var(--white);
}

.btn-info {
    background: linear-gradient(135deg, var(--info), #0891b2);
    color: var(--white);
    box-shadow: var(--shadow);
}

.btn-info:hover {
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: var(--white);
}

/* Outline buttons */
.btn-outline-primary {
    background: transparent;
    color: var(--primary);
    border: 2px solid var(--primary);
}

.btn-outline-primary:hover {
    background: var(--primary);
    color: var(--white);
    transform: translateY(-1px);
}

.btn-outline-secondary {
    background: transparent;
    color: var(--gray-600);
    border: 2px solid var(--gray-300);
}

.btn-outline-secondary:hover {
    background: var(--gray-600);
    color: var(--white);
    border-color: var(--gray-600);
}

/* Button Groups */
.btn-group {
    display: flex;
    gap: var(--spacing-2);
}

.btn-group .btn {
    margin: 0;
}

/* ================================
   Badges & Status Indicators
   ================================ */

.badge {
    display: inline-flex;
    align-items: center;
    gap: var(--spacing-1);
    padding: var(--spacing-1) var(--spacing-3);
    font-size: var(--font-size-xs);
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    border-radius: var(--radius-full);
    white-space: nowrap;
}

.badge-primary {
    background: var(--primary-light);
    color: var(--primary);
}

.badge-secondary {
    background: var(--gray-200);
    color: var(--gray-700);
}

.badge-success {
    background: var(--success-light);
    color: var(--success);
}

.badge-warning {
    background: var(--warning-light);
    color: var(--warning);
}

.badge-danger {
    background: var(--danger-light);
    color: var(--danger);
}

.badge-info {
    background: var(--info-light);
    color: var(--info);
}

/* Status badges with dots */
.badge-dot {
    position: relative;
    padding-left: var(--spacing-4);
}

.badge-dot::before {
    content: '';
    position: absolute;
    left: var(--spacing-2);
    top: 50%;
    transform: translateY(-50%);
    width: 6px;
    height: 6px;
    border-radius: 50%;
    background: currentColor;
}

/* ================================
   Tables
   ================================ */

.table-container {
    background: var(--white);
    border-radius: var(--radius-lg);
    overflow: hidden;
    box-shadow: var(--shadow);
    border: 1px solid var(--gray-200);
}

.table {
    width: 100%;
    border-collapse: collapse;
    margin: 0;
    font-size: var(--font-size-sm);
}

.table thead {
    background: linear-gradient(135deg, var(--gray-50), var(--gray-100));
}

.table thead th {
    padding: var(--spacing-4) var(--spacing-4);
    text-align: left;
    font-weight: 600;
    font-size: var(--font-size-xs);
    text-transform: uppercase;
    letter-spacing: 0.05em;
    color: var(--gray-700);
    border-bottom: 2px solid var(--gray-200);
    white-space: nowrap;
}

.table tbody td {
    padding: var(--spacing-4) var(--spacing-4);
    border-bottom: 1px solid var(--gray-100);
    vertical-align: middle;
}

.table tbody tr {
    transition: all 0.2s ease;
}

.table tbody tr:hover {
    background: var(--gray-50);
}

.table tbody tr:last-child td {
    border-bottom: none;
}

/* Table variants */
.table-striped tbody tr:nth-child(even) {
    background: var(--gray-50);
}

.table-striped tbody tr:nth-child(even):hover {
    background: var(--gray-100);
}

/* Table responsive */
.table-responsive {
    overflow-x: auto;
    border-radius: var(--radius-lg);
}

/* ================================
   Forms - Bootstrap 4 Compatible
   ================================ */

.form-group {
    margin-bottom: var(--spacing-4);
}

.form-control-label,
.col-form-label,
label {
    display: block;
    margin-bottom: var(--spacing-2);
    font-weight: 500;
    color: var(--gray-700);
    font-size: var(--font-size-sm);
}

/* Base form control styling */
.form-control {
    display: block;
    width: 100%;
    height: calc(1.5em + 0.75rem + 2px);
    padding: var(--spacing-3) var(--spacing-4);
    font-size: var(--font-size-sm);
    font-weight: 400;
    line-height: 1.5;
    color: var(--gray-700);
    background-color: var(--white);
    background-clip: padding-box;
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
    transition: all 0.15s ease-in-out;
}

.form-control:focus {
    color: var(--gray-700);
    background-color: var(--white);
    border-color: var(--primary);
    outline: 0;
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
}

.form-control::placeholder {
    color: var(--gray-400);
    opacity: 1;
}

.form-control:disabled,
.form-control[readonly] {
    background-color: var(--gray-100);
    border-color: var(--gray-200);
    opacity: 0.6;
    cursor: not-allowed;
}

/* Select specific styling for Bootstrap 4 */
select.form-control:not([size]):not([multiple]) {
    height: calc(1.5em + 0.75rem + 2px);
    appearance: none;
    -webkit-appearance: none;
    -moz-appearance: none;
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
    background-repeat: no-repeat;
    background-position: right 0.75rem center;
    background-size: 16px 12px;
    padding-right: 2.25rem;
    cursor: pointer;
}

select.form-control:focus {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%233b82f6' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
}

/* Select styling - Fixed for selected text display issue */
select.form-control:not([size]):not([multiple]) {
    height: calc(1.5em + 0.75rem + 4px);
    min-height: 38px;
    appearance: none;
    -webkit-appearance: none;
    -moz-appearance: none;
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
    background-repeat: no-repeat;
    background-position: right 0.75rem center;
    background-size: 16px 12px;
    padding: 0.5rem 2.25rem 0.5rem 0.75rem;
    cursor: pointer;
    line-height: 1.3;
    vertical-align: middle;
    text-overflow: ellipsis;
    white-space: nowrap;
    overflow: hidden;
}

select.form-control-sm:not([size]):not([multiple]) {
    height: calc(1.5em + 0.5rem + 4px);
    min-height: 32px;
    padding: 0.375rem 1.75rem 0.375rem 0.5rem;
    font-size: 0.875rem;
    background-size: 14px 10px;
    background-position: right 0.5rem center;
    line-height: 1.2;
}

select.form-control-lg:not([size]):not([multiple]) {
    height: calc(1.5em + 1rem + 4px);
    min-height: 48px;
    padding: 0.75rem 3rem 0.75rem 1rem;
    font-size: 1.125rem;
    background-size: 18px 14px;
    background-position: right 1rem center;
    line-height: 1.4;
}

select.form-control:focus {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%233b82f6' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
}

/* Select options styling */
select.form-control option {
    padding: 0.5rem 0.75rem;
    color: var(--gray-700);
    background: var(--white);
    line-height: 1.4;
    min-height: auto;
    height: auto;
}

select.form-control option:checked {
    background-color: var(--primary-light);
    color: var(--primary);
    font-weight: 500;
}

/* Custom select for Bootstrap 4 - Enhanced with better text display */
.custom-select {
    display: inline-block;
    width: 100%;
    height: calc(1.5em + 0.75rem + 4px);
    min-height: 38px;
    padding: 0.5rem 2.25rem 0.5rem 0.75rem;
    font-size: 0.875rem;
    font-weight: 400;
    line-height: 1.3;
    color: var(--gray-700);
    vertical-align: middle;
    background: var(--white) url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e") no-repeat right 0.75rem center/16px 12px;
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
    appearance: none;
    -webkit-appearance: none;
    -moz-appearance: none;
    cursor: pointer;
    transition: all 0.15s ease-in-out;
    text-overflow: ellipsis;
    white-space: nowrap;
    overflow: hidden;
}

.custom-select:focus {
    border-color: var(--primary);
    outline: 0;
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%233b82f6' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
}

.custom-select:disabled {
    color: var(--gray-400);
    background-color: var(--gray-100);
    border-color: var(--gray-200);
    cursor: not-allowed;
}

.custom-select-sm {
    height: calc(1.5em + 0.5rem + 4px);
    min-height: 32px;
    padding: 0.375rem 1.75rem 0.375rem 0.5rem;
    font-size: 0.875rem;
    background-size: 14px 10px;
    background-position: right 0.5rem center;
    line-height: 1.2;
}

.custom-select-lg {
    height: calc(1.5em + 1rem + 4px);
    min-height: 48px;
    padding: 0.75rem 3rem 0.75rem 1rem;
    font-size: 1.125rem;
    background-size: 18px 14px;
    background-position: right 1rem center;
    line-height: 1.4;
}

/* Filter form specific styling - Enhanced with proper text display */
.filter-select,
.filter-input {
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
    padding: 0.5rem 0.75rem;
    font-size: 0.875rem;
    transition: all 0.15s ease-in-out;
    background-color: var(--white);
    color: var(--gray-700);
    width: 100%;
    height: calc(1.5em + 0.75rem + 4px);
    min-height: 38px;
    line-height: 1.3;
    vertical-align: middle;
    text-overflow: ellipsis;
    white-space: nowrap;
    overflow: hidden;
}

.filter-select {
    appearance: none;
    -webkit-appearance: none;
    -moz-appearance: none;
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
    background-position: right 0.75rem center;
    background-repeat: no-repeat;
    background-size: 16px 12px;
    padding-right: 2.25rem;
    cursor: pointer;
}

.filter-select:focus,
.filter-input:focus {
    border-color: var(--primary);
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.15);
    outline: none;
}

.filter-select:focus {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%233b82f6' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
}

/* Base form control styling - Enhanced for better text display */
.form-control {
    display: block;
    width: 100%;
    height: calc(1.5em + 0.75rem + 4px);
    min-height: 38px;
    padding: 0.5rem 0.75rem;
    font-size: 0.875rem;
    font-weight: 400;
    line-height: 1.3;
    color: var(--gray-700);
    background-color: var(--white);
    background-clip: padding-box;
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
    transition: all 0.15s ease-in-out;
    vertical-align: middle;
}

.form-control:focus {
    color: var(--gray-700);
    background-color: var(--white);
    border-color: var(--primary);
    outline: 0;
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
}

.form-control::placeholder {
    color: var(--gray-400);
    opacity: 1;
}

.form-control:disabled,
.form-control[readonly] {
    background-color: var(--gray-100);
    border-color: var(--gray-200);
    opacity: 0.6;
    cursor: not-allowed;
}

/* Form control sizes with proper text display */
.form-control-sm,
.custom-select-sm {
    height: calc(1.5em + 0.5rem + 4px);
    min-height: 32px;
    padding: 0.375rem 0.5rem;
    font-size: 0.875rem;
    border-radius: calc(var(--radius) - 1px);
    line-height: 1.2;
}

.form-control-lg,
.custom-select-lg {
    height: calc(1.5em + 1rem + 4px);
    min-height: 48px;
    padding: 0.75rem 1rem;
    font-size: 1.125rem;
    border-radius: calc(var(--radius) + 2px);
    line-height: 1.4;
}

/* Additional fixes for specific browsers */
select.form-control {
    -moz-appearance: none;
    -webkit-appearance: none;
    appearance: none;
}

select.form-control::-ms-expand {
    display: none;
}

/* Firefox specific fixes */
@-moz-document url-prefix() {
    select.form-control {
        text-indent: 0.01px;
        text-overflow: '';
    }
}

/* Safari specific fixes */
@media screen and (-webkit-min-device-pixel-ratio: 0) {
    select.form-control {
        background-position: right 0.75rem center;
    }
}

/* Custom select for Bootstrap 4 - Enhanced */
.custom-select {
    display: inline-block;
    width: 100%;
    height: calc(1.5em + 0.75rem + 4px);
    padding: 0.375rem 2.25rem 0.375rem 0.75rem;
    font-size: 0.875rem;
    font-weight: 400;
    line-height: 1.5;
    color: var(--gray-700);
    vertical-align: middle;
    background: var(--white) url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e") no-repeat right 0.75rem center/16px 12px;
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
    appearance: none;
    -webkit-appearance: none;
    -moz-appearance: none;
    cursor: pointer;
    transition: all 0.15s ease-in-out;
}

.custom-select:focus {
    border-color: var(--primary);
    outline: 0;
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%233b82f6' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
}

.custom-select:disabled {
    color: var(--gray-400);
    background-color: var(--gray-100);
    border-color: var(--gray-200);
    cursor: not-allowed;
}

.custom-select-sm {
    height: calc(1.5em + 0.5rem + 4px);
    padding-top: 0.25rem;
    padding-bottom: 0.25rem;
    padding-left: 0.5rem;
    padding-right: 1.75rem;
    font-size: 0.75rem;
    background-size: 14px 10px;
    background-position: right 0.5rem center;
}

.custom-select-lg {
    height: calc(1.5em + 1rem + 4px);
    padding-top: 0.5rem;
    padding-bottom: 0.5rem;
    padding-left: 1rem;
    padding-right: 3rem;
    font-size: 1.125rem;
    background-size: 18px 14px;
    background-position: right 1rem center;
}

/* Custom file input - Enhanced for better appearance */
.custom-file {
    position: relative;
    display: inline-block;
    width: 100%;
    height: calc(1.5em + 0.75rem + 4px);
    margin-bottom: 0;
}

.custom-file-input {
    position: relative;
    z-index: 2;
    width: 100%;
    height: calc(1.5em + 0.75rem + 4px);
    margin: 0;
    opacity: 0;
    cursor: pointer;
}

.custom-file-input:focus ~ .custom-file-label {
    border-color: var(--primary);
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
}

.custom-file-input[disabled] ~ .custom-file-label,
.custom-file-input:disabled ~ .custom-file-label {
    background-color: var(--gray-100);
    cursor: not-allowed;
}

.custom-file-input:lang(en) ~ .custom-file-label::after {
    content: "Browse";
}

.custom-file-input ~ .custom-file-label[data-browse]::after {
    content: attr(data-browse);
}

.custom-file-label {
    position: absolute;
    top: 0;
    right: 0;
    left: 0;
    z-index: 1;
    height: calc(1.5em + 0.75rem + 4px);
    padding: 0.375rem 0.75rem;
    font-weight: 400;
    line-height: 1.5;
    color: var(--gray-600);
    background-color: var(--white);
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.custom-file-label::after {
    position: absolute;
    top: 0;
    right: 0;
    bottom: 0;
    z-index: 3;
    display: block;
    height: calc(1.5em + 0.75rem);
    padding: 0.375rem 0.75rem;
    line-height: 1.5;
    color: var(--gray-600);
    content: "Browse";
    background-color: var(--gray-100);
    border-left: 2px solid var(--gray-300);
    border-radius: 0 calc(var(--radius) - 2px) calc(var(--radius) - 2px) 0;
    cursor: pointer;
    font-size: 0.875rem;
    font-weight: 500;
}

/* File input with "Choose File" text styling */
input[type="file"] {
    display: block;
    width: 100%;
    height: calc(1.5em + 0.75rem + 4px);
    padding: 0.375rem 0.75rem;
    font-size: 0.875rem;
    font-weight: 400;
    line-height: 1.5;
    color: var(--gray-700);
    background-color: var(--white);
    background-clip: padding-box;
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
    transition: all 0.15s ease-in-out;
    cursor: pointer;
}

input[type="file"]:focus {
    color: var(--gray-700);
    background-color: var(--white);
    border-color: var(--primary);
    outline: 0;
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
}

input[type="file"]:disabled {
    background-color: var(--gray-100);
    border-color: var(--gray-200);
    opacity: 0.6;
    cursor: not-allowed;
}

/* File input button styling */
input[type="file"]::-webkit-file-upload-button {
    padding: 0.375rem 0.75rem;
    margin: -0.375rem -0.75rem -0.375rem 0.75rem;
    color: var(--gray-600);
    background-color: var(--gray-100);
    border: none;
    border-left: 2px solid var(--gray-300);
    border-radius: 0;
    cursor: pointer;
    font-size: 0.875rem;
    font-weight: 500;
    transition: all 0.15s ease-in-out;
}

input[type="file"]::-webkit-file-upload-button:hover {
    background-color: var(--gray-200);
    color: var(--gray-700);
}

input[type="file"]::-moz-file-upload-button {
    padding: 0.375rem 0.75rem;
    margin: -0.375rem -0.75rem -0.375rem 0.75rem;
    color: var(--gray-600);
    background-color: var(--gray-100);
    border: none;
    border-left: 2px solid var(--gray-300);
    cursor: pointer;
    font-size: 0.875rem;
    font-weight: 500;
    transition: all 0.15s ease-in-out;
}

input[type="file"]::-moz-file-upload-button:hover {
    background-color: var(--gray-200);
    color: var(--gray-700);
}

/* Filter form specific styling - Enhanced */
.filter-select,
.filter-input {
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
    padding: 0.375rem 0.75rem;
    font-size: 0.875rem;
    transition: all 0.15s ease-in-out;
    background-color: var(--white);
    color: var(--gray-700);
    width: 100%;
    height: calc(1.5em + 0.75rem + 4px);
    line-height: 1.5;
}

.filter-select {
    appearance: none;
    -webkit-appearance: none;
    -moz-appearance: none;
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
    background-position: right 0.75rem center;
    background-repeat: no-repeat;
    background-size: 16px 12px;
    padding-right: 2.25rem;
    cursor: pointer;
}

.filter-select:focus,
.filter-input:focus {
    border-color: var(--primary);
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.15);
    outline: none;
}

.filter-select:focus {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%233b82f6' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
}

/* Specific fixes for document upload styling */
.document-upload-section {
    background: var(--white);
    border: 1px solid var(--gray-200);
    border-radius: var(--radius-md);
    padding: var(--spacing-4);
    margin-bottom: var(--spacing-4);
}

.document-upload-section .form-group {
    margin-bottom: var(--spacing-3);
}

.document-upload-section .custom-file-label {
    color: var(--gray-500);
    font-size: 0.875rem;
}

.document-upload-section .custom-file-label::after {
    content: "Pilih File";
    background-color: var(--primary-light);
    color: var(--primary);
    border-color: var(--primary);
    font-weight: 500;
}

.document-upload-section .custom-file-input:focus ~ .custom-file-label::after {
    background-color: var(--primary);
    color: var(--white);
}

/* Enhanced styling for form with file uploads */
.form-section {
    background: var(--white);
    border-radius: var(--radius-lg);
    padding: var(--spacing-6);
    margin-bottom: var(--spacing-6);
    box-shadow: var(--shadow);
    border: 1px solid var(--gray-200);
}

.form-section .form-group label {
    font-weight: 500;
    color: var(--gray-700);
    margin-bottom: var(--spacing-2);
    font-size: 0.875rem;
}

/* View/Lihat button styling */
.btn-view-file {
    display: inline-flex;
    align-items: center;
    gap: var(--spacing-1);
    padding: var(--spacing-1) var(--spacing-3);
    background: var(--info-light);
    color: var(--info);
    border: 1px solid var(--info);
    border-radius: var(--radius);
    text-decoration: none;
    font-size: 0.75rem;
    font-weight: 500;
    transition: all 0.2s ease;
}

.btn-view-file:hover {
    background: var(--info);
    color: var(--white);
    text-decoration: none;
    transform: translateY(-1px);
}

/* PDF indicator styling */
.file-type-indicator {
    display: inline-flex;
    align-items: center;
    gap: var(--spacing-1);
    font-size: 0.75rem;
    color: var(--gray-500);
    margin-left: var(--spacing-2);
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .filter-select,
    .filter-input,
    .form-control,
    .custom-select {
        font-size: 0.8125rem;
        height: calc(1.5em + 0.5rem + 4px);
        padding: 0.25rem 0.5rem;
    }

    select.form-control:not([size]):not([multiple]) {
        padding-right: 1.75rem;
        background-size: 14px 10px;
        background-position: right 0.5rem center;
    }

    .custom-file-label {
        font-size: 0.8125rem;
        height: calc(1.5em + 0.5rem + 4px);
        padding: 0.25rem 0.5rem;
    }

    .custom-file-label::after {
        font-size: 0.75rem;
        padding: 0.25rem 0.5rem;
        height: calc(1.5em + 0.5rem);
    }
}

/* Form control sizes - Bootstrap 4 compatible */
.form-control-sm,
.custom-select-sm {
    height: calc(1.5em + 0.5rem + 2px);
    padding: var(--spacing-2) var(--spacing-3);
    font-size: var(--font-size-xs);
    border-radius: calc(var(--radius) - 1px);
}

select.form-control-sm:not([size]):not([multiple]),
.custom-select-sm {
    height: calc(1.5em + 0.5rem + 2px);
    padding-right: 1.75rem;
    background-size: 14px 10px;
}

.form-control-lg,
.custom-select-lg {
    height: calc(1.5em + 1rem + 2px);
    padding: var(--spacing-4) var(--spacing-5);
    font-size: var(--font-size-lg);
    border-radius: calc(var(--radius) + 2px);
}

select.form-control-lg:not([size]):not([multiple]),
.custom-select-lg {
    height: calc(1.5em + 1rem + 2px);
    padding-right: 3rem;
    background-size: 18px 14px;
}

/* Filter form specific styling */
.filter-select,
.filter-input {
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
    padding: var(--spacing-3) var(--spacing-4);
    font-size: var(--font-size-sm);
    transition: all 0.3s ease;
    background-color: var(--white);
    color: var(--gray-700);
    width: 100%;
    height: calc(1.5em + 0.75rem + 2px);
}

.filter-select {
    appearance: none;
    -webkit-appearance: none;
    -moz-appearance: none;
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
    background-position: right 0.75rem center;
    background-repeat: no-repeat;
    background-size: 16px 12px;
    padding-right: 2.25rem;
    cursor: pointer;
}

.filter-select:focus,
.filter-input:focus {
    border-color: var(--primary);
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.15);
    outline: none;
}

.filter-select:focus {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%233b82f6' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
}

/* Input group styling for Bootstrap 4 */
.input-group {
    position: relative;
    display: flex;
    flex-wrap: wrap;
    align-items: stretch;
    width: 100%;
}

.input-group > .form-control,
.input-group > .form-control-plaintext,
.input-group > .custom-select,
.input-group > .custom-file {
    position: relative;
    flex: 1 1 auto;
    width: 1%;
    min-width: 0;
    margin-bottom: 0;
}

.input-group > .form-control + .form-control,
.input-group > .form-control + .custom-select,
.input-group > .form-control + .custom-file,
.input-group > .form-control-plaintext + .form-control,
.input-group > .form-control-plaintext + .custom-select,
.input-group > .form-control-plaintext + .custom-file,
.input-group > .custom-select + .form-control,
.input-group > .custom-select + .custom-select,
.input-group > .custom-select + .custom-file,
.input-group > .custom-file + .form-control,
.input-group > .custom-file + .custom-select,
.input-group > .custom-file + .custom-file {
    margin-left: -2px;
}

.input-group > .form-control:focus,
.input-group > .custom-select:focus,
.input-group > .custom-file .custom-file-input:focus ~ .custom-file-label {
    z-index: 3;
}

.input-group > .custom-file .custom-file-input:focus {
    z-index: 4;
}

.input-group > .form-control:not(:last-child),
.input-group > .custom-select:not(:last-child) {
    border-top-right-radius: 0;
    border-bottom-right-radius: 0;
}

.input-group > .form-control:not(:first-child),
.input-group > .custom-select:not(:first-child) {
    border-top-left-radius: 0;
    border-bottom-left-radius: 0;
}

.input-group > .custom-file {
    display: flex;
    align-items: center;
}

.input-group > .custom-file:not(:last-child) .custom-file-label,
.input-group > .custom-file:not(:last-child) .custom-file-label::after {
    border-top-right-radius: 0;
    border-bottom-right-radius: 0;
}

.input-group > .custom-file:not(:first-child) .custom-file-label {
    border-top-left-radius: 0;
    border-bottom-left-radius: 0;
}

.input-group-prepend,
.input-group-append {
    display: flex;
}

.input-group-prepend .btn,
.input-group-append .btn {
    position: relative;
    z-index: 2;
}

.input-group-prepend .btn:focus,
.input-group-append .btn:focus {
    z-index: 3;
}

.input-group-prepend .btn + .btn,
.input-group-prepend .btn + .input-group-text,
.input-group-prepend .input-group-text + .input-group-text,
.input-group-prepend .input-group-text + .btn,
.input-group-append .btn + .btn,
.input-group-append .btn + .input-group-text,
.input-group-append .input-group-text + .input-group-text,
.input-group-append .input-group-text + .btn {
    margin-left: -2px;
}

.input-group-prepend {
    margin-right: -2px;
}

.input-group-append {
    margin-left: -2px;
}

.input-group-text {
    display: flex;
    align-items: center;
    padding: var(--spacing-3) var(--spacing-4);
    margin-bottom: 0;
    font-size: var(--font-size-sm);
    font-weight: 400;
    line-height: 1.5;
    color: var(--gray-600);
    text-align: center;
    white-space: nowrap;
    background-color: var(--gray-100);
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
}

.input-group-text input[type="radio"],
.input-group-text input[type="checkbox"] {
    margin-top: 0;
}

.input-group-lg > .form-control:not(textarea),
.input-group-lg > .custom-select {
    height: calc(1.5em + 1rem + 2px);
}

.input-group-lg > .form-control,
.input-group-lg > .custom-select,
.input-group-lg > .input-group-prepend > .input-group-text,
.input-group-lg > .input-group-append > .input-group-text,
.input-group-lg > .input-group-prepend > .btn,
.input-group-lg > .input-group-append > .btn {
    padding: var(--spacing-4) var(--spacing-5);
    font-size: var(--font-size-lg);
    border-radius: calc(var(--radius) + 2px);
}

.input-group-sm > .form-control:not(textarea),
.input-group-sm > .custom-select {
    height: calc(1.5em + 0.5rem + 2px);
}

.input-group-sm > .form-control,
.input-group-sm > .custom-select,
.input-group-sm > .input-group-prepend > .input-group-text,
.input-group-sm > .input-group-append > .input-group-text,
.input-group-sm > .input-group-prepend > .btn,
.input-group-sm > .input-group-append > .btn {
    padding: var(--spacing-2) var(--spacing-3);
    font-size: var(--font-size-xs);
    border-radius: calc(var(--radius) - 1px);
}

.input-group-lg > .custom-select,
.input-group-sm > .custom-select {
    padding-right: 1.75rem;
}

/* Validation states for Bootstrap 4 */
.was-validated .form-control:valid,
.form-control.is-valid {
    border-color: var(--success);
    padding-right: calc(1.5em + 0.75rem);
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' width='8' height='8' viewBox='0 0 8 8'%3e%3cpath fill='%2310b981' d='m2.3 6.73 0.13-0.12 2.19-2.19 1.64 1.64 0.13 0.13 0.88-0.88-0.13-0.13-2.52-2.52-0.13-0.13-1.01 1.01 0.13 0.13 1.64 1.64z'/%3e%3c/svg%3e");
    background-repeat: no-repeat;
    background-position: right calc(0.375em + 0.1875rem) center;
    background-size: calc(0.75em + 0.375rem) calc(0.75em + 0.375rem);
}

.was-validated .form-control:valid:focus,
.form-control.is-valid:focus {
    border-color: var(--success);
    box-shadow: 0 0 0 0.2rem rgba(16, 185, 129, 0.25);
}

.was-validated .form-control:invalid,
.form-control.is-invalid {
    border-color: var(--danger);
    padding-right: calc(1.5em + 0.75rem);
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' width='12' height='12' fill='none' stroke='%23ef4444' viewBox='0 0 12 12'%3e%3ccircle cx='6' cy='6' r='4.5'/%3e%3cpath d='m5.8 4.6 2.4 2.4m0-2.4L5.8 7'/%3e%3c/svg%3e");
    background-repeat: no-repeat;
    background-position: right calc(0.375em + 0.1875rem) center;
    background-size: calc(0.75em + 0.375rem) calc(0.75em + 0.375rem);
}

.was-validated .form-control:invalid:focus,
.form-control.is-invalid:focus {
    border-color: var(--danger);
    box-shadow: 0 0 0 0.2rem rgba(239, 68, 68, 0.25);
}

/* Custom select validation */
.was-validated .custom-select:valid,
.custom-select.is-valid {
    border-color: var(--success);
    padding-right: calc(0.75em + 2.3125rem);
    background: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e") no-repeat right 0.75rem center/16px 12px, url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' width='8' height='8' viewBox='0 0 8 8'%3e%3cpath fill='%2310b981' d='m2.3 6.73 0.13-0.12 2.19-2.19 1.64 1.64 0.13 0.13 0.88-0.88-0.13-0.13-2.52-2.52-0.13-0.13-1.01 1.01 0.13 0.13 1.64 1.64z'/%3e%3c/svg%3e") var(--white) no-repeat center right 1.75rem/calc(0.75em + 0.375rem) calc(0.75em + 0.375rem);
}

.was-validated .custom-select:valid:focus,
.custom-select.is-valid:focus {
    border-color: var(--success);
    box-shadow: 0 0 0 0.2rem rgba(16, 185, 129, 0.25);
}

.was-validated .custom-select:invalid,
.custom-select.is-invalid {
    border-color: var(--danger);
    padding-right: calc(0.75em + 2.3125rem);
    background: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e") no-repeat right 0.75rem center/16px 12px, url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' width='12' height='12' fill='none' stroke='%23ef4444' viewBox='0 0 12 12'%3e%3ccircle cx='6' cy='6' r='4.5'/%3e%3cpath d='m5.8 4.6 2.4 2.4m0-2.4L5.8 7'/%3e%3c/svg%3e") var(--white) no-repeat center right 1.75rem/calc(0.75em + 0.375rem) calc(0.75em + 0.375rem);
}

.was-validated .custom-select:invalid:focus,
.custom-select.is-invalid:focus {
    border-color: var(--danger);
    box-shadow: 0 0 0 0.2rem rgba(239, 68, 68, 0.25);
}

/* Feedback text */
.valid-feedback {
    display: none;
    width: 100%;
    margin-top: 0.25rem;
    font-size: var(--font-size-sm);
    color: var(--success);
}

.valid-tooltip {
    position: absolute;
    top: 100%;
    z-index: 5;
    display: none;
    max-width: 100%;
    padding: 0.25rem 0.5rem;
    margin-top: 0.1rem;
    font-size: var(--font-size-xs);
    color: var(--white);
    background-color: rgba(16, 185, 129, 0.9);
    border-radius: var(--radius);
}

.was-validated :valid ~ .valid-feedback,
.was-validated :valid ~ .valid-tooltip,
.is-valid ~ .valid-feedback,
.is-valid ~ .valid-tooltip {
    display: block;
}

.invalid-feedback {
    display: none;
    width: 100%;
    margin-top: 0.25rem;
    font-size: var(--font-size-sm);
    color: var(--danger);
}

.invalid-tooltip {
    position: absolute;
    top: 100%;
    z-index: 5;
    display: none;
    max-width: 100%;
    padding: 0.25rem 0.5rem;
    margin-top: 0.1rem;
    font-size: var(--font-size-xs);
    color: var(--white);
    background-color: rgba(239, 68, 68, 0.9);
    border-radius: var(--radius);
}

.was-validated :invalid ~ .invalid-feedback,
.was-validated :invalid ~ .invalid-tooltip,
.is-invalid ~ .invalid-feedback,
.is-invalid ~ .invalid-tooltip {
    display: block;
}

/* Custom file input for Bootstrap 4 */
.custom-file {
    position: relative;
    display: inline-block;
    width: 100%;
    height: calc(1.5em + 0.75rem + 2px);
    margin-bottom: 0;
}

.custom-file-input {
    position: relative;
    z-index: 2;
    width: 100%;
    height: calc(1.5em + 0.75rem + 2px);
    margin: 0;
    opacity: 0;
}

.custom-file-input:focus ~ .custom-file-label {
    border-color: var(--primary);
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
}

.custom-file-input[disabled] ~ .custom-file-label,
.custom-file-input:disabled ~ .custom-file-label {
    background-color: var(--gray-100);
}

.custom-file-input:lang(en) ~ .custom-file-label::after {
    content: "Browse";
}

.custom-file-input ~ .custom-file-label[data-browse]::after {
    content: attr(data-browse);
}

.custom-file-label {
    position: absolute;
    top: 0;
    right: 0;
    left: 0;
    z-index: 1;
    height: calc(1.5em + 0.75rem + 2px);
    padding: var(--spacing-3) var(--spacing-4);
    font-weight: 400;
    line-height: 1.5;
    color: var(--gray-600);
    background-color: var(--white);
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
}

.custom-file-label::after {
    position: absolute;
    top: 0;
    right: 0;
    bottom: 0;
    z-index: 3;
    display: block;
    height: calc(1.5em + 0.75rem);
    padding: var(--spacing-3) var(--spacing-4);
    line-height: 1.5;
    color: var(--gray-600);
    content: "Browse";
    background-color: var(--gray-100);
    border-left: inherit;
    border-radius: 0 calc(var(--radius) - 1px) calc(var(--radius) - 1px) 0;
}

/* Checkbox and Radio custom styling for Bootstrap 4 */
.custom-control {
    position: relative;
    display: block;
    min-height: 1.5rem;
    padding-left: 1.5rem;
}

.custom-control-inline {
    display: inline-flex;
    margin-right: 1rem;
}

.custom-control-input {
    position: absolute;
    left: 0;
    z-index: -1;
    width: 1rem;
    height: 1.25rem;
    opacity: 0;
}

.custom-control-input:checked ~ .custom-control-label::before {
    color: var(--white);
    border-color: var(--primary);
    background-color: var(--primary);
}

.custom-control-input:focus ~ .custom-control-label::before {
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
}

.custom-control-input:focus:not(:checked) ~ .custom-control-label::before {
    border-color: var(--primary);
}

.custom-control-input:not(:disabled):active ~ .custom-control-label::before {
    color: var(--white);
    background-color: var(--primary-light);
    border-color: var(--primary-light);
}

.custom-control-input[disabled] ~ .custom-control-label,
.custom-control-input:disabled ~ .custom-control-label {
    color: var(--gray-400);
}

.custom-control-input[disabled] ~ .custom-control-label::before,
.custom-control-input:disabled ~ .custom-control-label::before {
    background-color: var(--gray-100);
}

.custom-control-label {
    position: relative;
    margin-bottom: 0;
    vertical-align: top;
}

.custom-control-label::before {
    position: absolute;
    top: 0.25rem;
    left: -1.5rem;
    display: block;
    width: 1rem;
    height: 1rem;
    pointer-events: none;
    content: "";
    background-color: var(--white);
    border: 2px solid var(--gray-400);
}

.custom-control-label::after {
    position: absolute;
    top: 0.25rem;
    left: -1.5rem;
    display: block;
    width: 1rem;
    height: 1rem;
    content: "";
    background: no-repeat 50% / 50% 50%;
}

.custom-checkbox .custom-control-label::before {
    border-radius: var(--radius-sm);
}

.custom-checkbox .custom-control-input:checked ~ .custom-control-label::after {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' width='8' height='8' viewBox='0 0 8 8'%3e%3cpath fill='%23fff' d='m6.564.75-3.59 3.612-1.538-1.55L0 4.26 2.974 7.25 8 2.193z'/%3e%3c/svg%3e");
}

.custom-checkbox .custom-control-input:indeterminate ~ .custom-control-label::before {
    border-color: var(--primary);
    background-color: var(--primary);
}

.custom-checkbox .custom-control-input:indeterminate ~ .custom-control-label::after {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' width='4' height='4' viewBox='0 0 4 4'%3e%3cpath stroke='%23fff' d='m0 2h4'/%3e%3c/svg%3e");
}

.custom-checkbox .custom-control-input:disabled:checked ~ .custom-control-label::before {
    background-color: rgba(59, 130, 246, 0.5);
}

.custom-checkbox .custom-control-input:disabled:indeterminate ~ .custom-control-label::before {
    background-color: rgba(59, 130, 246, 0.5);
}

.custom-radio .custom-control-label::before {
    border-radius: 50%;
}

.custom-radio .custom-control-input:checked ~ .custom-control-label::after {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' width='12' height='12' viewBox='-4 -4 8 8'%3e%3ccircle r='3' fill='%23fff'/%3e%3c/svg%3e");
}

.custom-radio .custom-control-input:disabled:checked ~ .custom-control-label::before {
    background-color: rgba(59, 130, 246, 0.5);
}

.custom-switch {
    padding-left: 2.25rem;
}

.custom-switch .custom-control-label::before {
    left: -2.25rem;
    width: 1.75rem;
    pointer-events: all;
    border-radius: 0.5rem;
}

.custom-switch .custom-control-label::after {
    top: calc(0.25rem + 2px);
    left: calc(-2.25rem + 2px);
    width: calc(1rem - 4px);
    height: calc(1rem - 4px);
    background-color: var(--gray-400);
    border-radius: 0.5rem;
    transition: transform 0.15s ease-in-out, background-color 0.15s ease-in-out, border-color 0.15s ease-in-out, box-shadow 0.15s ease-in-out;
}

@media (prefers-reduced-motion: reduce) {
    .custom-switch .custom-control-label::after {
        transition: none;
    }
}

.custom-switch .custom-control-input:checked ~ .custom-control-label::after {
    background-color: var(--white);
    transform: translateX(0.75rem);
}

.custom-switch .custom-control-input:disabled:checked ~ .custom-control-label::before {
    background-color: rgba(59, 130, 246, 0.5);
}

/* Custom range for Bootstrap 4 */
.custom-range {
    width: 100%;
    height: 1.4rem;
    padding: 0;
    background-color: transparent;
    appearance: none;
    -webkit-appearance: none;
}

.custom-range:focus {
    outline: none;
}

.custom-range:focus::-webkit-slider-thumb {
    box-shadow: 0 0 0 1px var(--gray-50), 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
}

.custom-range:focus::-moz-range-thumb {
    box-shadow: 0 0 0 1px var(--gray-50), 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
}

.custom-range::-moz-focus-outer {
    border: 0;
}

.custom-range::-webkit-slider-track {
    width: 100%;
    height: 0.5rem;
    color: transparent;
    cursor: pointer;
    background-color: var(--gray-200);
    border-color: transparent;
    border-radius: 1rem;
}

.custom-range::-webkit-slider-thumb {
    width: 1rem;
    height: 1rem;
    margin-top: -0.25rem;
    background-color: var(--primary);
    border: 0;
    border-radius: 1rem;
    cursor: pointer;
    appearance: none;
    -webkit-appearance: none;
    transition: background-color 0.15s ease-in-out, border-color 0.15s ease-in-out, box-shadow 0.15s ease-in-out;
}

@media (prefers-reduced-motion: reduce) {
    .custom-range::-webkit-slider-thumb {
        transition: none;
    }
}

.custom-range::-webkit-slider-thumb:active {
    background-color: var(--primary-light);
}

.custom-range::-moz-range-track {
    width: 100%;
    height: 0.5rem;
    color: transparent;
    cursor: pointer;
    background-color: var(--gray-200);
    border-color: transparent;
    border-radius: 1rem;
}

.custom-range::-moz-range-thumb {
    width: 1rem;
    height: 1rem;
    background-color: var(--primary);
    border: 0;
    border-radius: 1rem;
    cursor: pointer;
    appearance: none;
    transition: background-color 0.15s ease-in-out, border-color 0.15s ease-in-out, box-shadow 0.15s ease-in-out;
}

@media (prefers-reduced-motion: reduce) {
    .custom-range::-moz-range-thumb {
        transition: none;
    }
}

.custom-range::-moz-range-thumb:active {
    background-color: var(--primary-light);
}

.custom-range:disabled::-webkit-slider-track {
    cursor: default;
}

.custom-range:disabled::-webkit-slider-thumb {
    background-color: var(--gray-400);
    cursor: default;
}

.custom-range:disabled::-moz-range-track {
    cursor: default;
}

.custom-range:disabled::-moz-range-thumb {
    background-color: var(--gray-400);
    cursor: default;
}0.2rem rgba(59, 130, 246, 0.25);
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%233b82f6' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
}

.form-select:disabled,
select.form-control:disabled,
.custom-select:disabled {
    background-color: var(--gray-100);
    border-color: var(--gray-200);
    opacity: 0.6;
    cursor: not-allowed;
}

/* Input groups */
.input-group {
    display: flex;
    align-items: stretch;
    width: 100%;
}

.input-group .form-control {
    position: relative;
    flex: 1 1 auto;
    width: 1%;
    min-width: 0;
    margin-bottom: 0;
}

.input-group .form-control:not(:last-child) {
    border-top-right-radius: 0;
    border-bottom-right-radius: 0;
}

.input-group .form-control:not(:first-child) {
    border-top-left-radius: 0;
    border-bottom-left-radius: 0;
}

.input-group-text {
    display: flex;
    align-items: center;
    padding: var(--spacing-3) var(--spacing-4);
    font-size: var(--font-size-sm);
    font-weight: 400;
    line-height: 1.5;
    color: var(--gray-600);
    text-align: center;
    white-space: nowrap;
    background: var(--gray-100);
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
}

/* ================================
   Alerts & Notifications
   ================================ */

.alert {
    position: relative;
    padding: var(--spacing-4) var(--spacing-5);
    margin-bottom: var(--spacing-4);
    border: 1px solid transparent;
    border-radius: var(--radius-md);
    font-size: var(--font-size-sm);
}

.alert-primary {
    color: var(--primary);
    background: var(--primary-light);
    border-color: var(--primary);
}

.alert-success {
    color: var(--success);
    background: var(--success-light);
    border-color: var(--success);
}

.alert-warning {
    color: var(--warning);
    background: var(--warning-light);
    border-color: var(--warning);
}

.alert-danger {
    color: var(--danger);
    background: var(--danger-light);
    border-color: var(--danger);
}

.alert-info {
    color: var(--info);
    background: var(--info-light);
    border-color: var(--info);
}

.alert-dismissible {
    padding-right: var(--spacing-10);
}

.alert .btn-close {
    position: absolute;
    top: 0;
    right: 0;
    z-index: 2;
    padding: var(--spacing-4) var(--spacing-5);
    background: none;
    border: none;
    color: inherit;
    opacity: 0.5;
    cursor: pointer;
}

.alert .btn-close:hover {
    opacity: 1;
}

/* ================================
   Modals
   ================================ */
/*
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(15, 23, 42, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1050;
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s ease;
    backdrop-filter: blur(4px);
}

.modal-overlay.show {
    opacity: 1;
    visibility: visible;
}

.modal {
    background: var(--white);
    border-radius: var(--radius-xl);
    max-width: 500px;
    width: 90%;
    max-height: 90vh;
    overflow: hidden;
    box-shadow: var(--shadow-xl);
    transform: scale(0.9);
    transition: all 0.3s ease;
}

.modal-overlay.show .modal {
    transform: scale(1);
}

.modal-header {
    padding: var(--spacing-6) var(--spacing-6) var(--spacing-4);
    border-bottom: 1px solid var(--gray-200);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-title {
    margin: 0;
    font-size: var(--font-size-xl);
    font-weight: 600;
    color: var(--gray-900);
}

.modal-body {
    padding: var(--spacing-6);
    overflow-y: auto;
}

.modal-footer {
    padding: var(--spacing-4) var(--spacing-6) var(--spacing-6);
    border-top: 1px solid var(--gray-200);
    display: flex;
    gap: var(--spacing-3);
    justify-content: flex-end;
} */

/* ================================
   Utilities
   ================================ */

/* Text utilities */
.text-primary { color: var(--primary) !important; }
.text-secondary { color: var(--gray-600) !important; }
.text-success { color: var(--success) !important; }
.text-warning { color: var(--warning) !important; }
.text-danger { color: var(--danger) !important; }
.text-info { color: var(--info) !important; }
.text-muted { color: var(--gray-500) !important; }

/* Background utilities */
.bg-primary { background-color: var(--primary) !important; }
.bg-secondary { background-color: var(--gray-100) !important; }
.bg-success { background-color: var(--success) !important; }
.bg-warning { background-color: var(--warning) !important; }
.bg-danger { background-color: var(--danger) !important; }
.bg-info { background-color: var(--info) !important; }
.bg-light { background-color: var(--gray-50) !important; }
.bg-dark { background-color: var(--gray-800) !important; }

/* Spacing utilities */
.m-0 { margin: 0 !important; }
.mt-0 { margin-top: 0 !important; }
.mb-0 { margin-bottom: 0 !important; }
.p-0 { padding: 0 !important; }
.pt-0 { padding-top: 0 !important; }
.pb-0 { padding-bottom: 0 !important; }

.m-1 { margin: var(--spacing-1) !important; }
.m-2 { margin: var(--spacing-2) !important; }
.m-3 { margin: var(--spacing-3) !important; }
.m-4 { margin: var(--spacing-4) !important; }

.p-1 { padding: var(--spacing-1) !important; }
.p-2 { padding: var(--spacing-2) !important; }
.p-3 { padding: var(--spacing-3) !important; }
.p-4 { padding: var(--spacing-4) !important; }

/* Display utilities */
.d-none { display: none !important; }
.d-block { display: block !important; }
.d-inline { display: inline !important; }
.d-inline-block { display: inline-block !important; }
.d-flex { display: flex !important; }

/* Flex utilities */
.justify-content-start { justify-content: flex-start !important; }
.justify-content-end { justify-content: flex-end !important; }
.justify-content-center { justify-content: center !important; }
.justify-content-between { justify-content: space-between !important; }
.align-items-start { align-items: flex-start !important; }
.align-items-end { align-items: flex-end !important; }
.align-items-center { align-items: center !important; }

/* Text alignment */
.text-left { text-align: left !important; }
.text-center { text-align: center !important; }
.text-right { text-align: right !important; }

/* Font weight */
.font-weight-light { font-weight: 300 !important; }
.font-weight-normal { font-weight: 400 !important; }
.font-weight-bold { font-weight: 700 !important; }

/* Borders */
.border { border: 1px solid var(--gray-200) !important; }
.border-top { border-top: 1px solid var(--gray-200) !important; }
.border-bottom { border-bottom: 1px solid var(--gray-200) !important; }
.border-0 { border: 0 !important; }

/* Border radius */
.rounded { border-radius: var(--radius) !important; }
.rounded-lg { border-radius: var(--radius-lg) !important; }
.rounded-full { border-radius: var(--radius-full) !important; }

/* Shadows */
.shadow-none { box-shadow: none !important; }
.shadow-sm { box-shadow: var(--shadow-sm) !important; }
.shadow { box-shadow: var(--shadow) !important; }
.shadow-md { box-shadow: var(--shadow-md) !important; }
.shadow-lg { box-shadow: var(--shadow-lg) !important; }

/* ================================
   Responsive Design
   ================================ */

@media (max-width: 768px) {
    :root {
        --spacing-4: 0.75rem;
        --spacing-5: 1rem;
        --spacing-6: 1.25rem;
    }

    .card-header,
    .card-body {
        padding: var(--spacing-4);
    }

    .btn {
        padding: var(--spacing-2) var(--spacing-3);
        font-size: var(--font-size-xs);
    }

    .table thead th,
    .table tbody td {
        padding: var(--spacing-2) var(--spacing-3);
        font-size: var(--font-size-xs);
    }

    .modal {
        margin: var(--spacing-4);
        width: calc(100% - var(--spacing-8));
    }

    .modal-body {
        padding: var(--spacing-4);
    }

    .modal-header,
    .modal-footer {
        padding: var(--spacing-4);
    }
}

/* ================================
   Custom Animations
   ================================ */

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

@keyframes slideIn {
    from { transform: translateX(-10px); opacity: 0; }
    to { transform: translateX(0); opacity: 1; }
}

@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.05); }
    100% { transform: scale(1); }
}

.animate-fadeIn {
    animation: fadeIn 0.3s ease-out;
}

.animate-slideIn {
    animation: slideIn 0.3s ease-out;
}

.animate-pulse {
    animation: pulse 2s infinite;
}

/* ================================
   Loading States
   ================================ */

.loading {
    position: relative;
    overflow: hidden;
    pointer-events: none;
}

.loading::after {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg,
        transparent,
        rgba(255, 255, 255, 0.6),
        transparent
    );
    animation: loading 1.5s infinite;
}

@keyframes loading {
    0% { left: -100%; }
    100% { left: 100%; }
}

.btn-loading {
    position: relative;
    color: transparent !important;
    pointer-events: none;
}

.btn-loading::before {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    width: 16px;
    height: 16px;
    margin: -8px 0 0 -8px;
    border: 2px solid transparent;
    border-top: 2px solid currentColor;
    border-radius: 50%;
    animation: spin 1s linear infinite;
    color: var(--white);
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* ================================
   Skeleton Loading
   ================================ */

.skeleton {
    background: linear-gradient(90deg,
        var(--gray-200) 25%,
        var(--gray-100) 50%,
        var(--gray-200) 75%
    );
    background-size: 200% 100%;
    animation: skeleton-loading 1.5s infinite;
    border-radius: var(--radius);
}

@keyframes skeleton-loading {
    0% { background-position: 200% 0; }
    100% { background-position: -200% 0; }
}

.skeleton-text {
    height: 1rem;
    margin-bottom: var(--spacing-2);
}

.skeleton-text:last-child {
    margin-bottom: 0;
    width: 60%;
}

.skeleton-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
}

.skeleton-button {
    height: 40px;
    width: 100px;
}

/* ================================
   Special Components
   ================================ */

/* Progress bars */
.progress {
    display: flex;
    height: 8px;
    overflow: hidden;
    background: var(--gray-200);
    border-radius: var(--radius-full);
}

.progress-bar {
    display: flex;
    flex-direction: column;
    justify-content: center;
    overflow: hidden;
    color: var(--white);
    text-align: center;
    white-space: nowrap;
    background: var(--primary);
    transition: width 0.6s ease;
    border-radius: var(--radius-full);
}

.progress-bar-success { background: var(--success); }
.progress-bar-warning { background: var(--warning); }
.progress-bar-danger { background: var(--danger); }
.progress-bar-info { background: var(--info); }

/* Breadcrumbs */
.breadcrumb {
    display: flex;
    flex-wrap: wrap;
    padding: 0;
    margin: 0;
    list-style: none;
    background: transparent;
}

.breadcrumb-item {
    display: flex;
    align-items: center;
}

.breadcrumb-item + .breadcrumb-item {
    padding-left: var(--spacing-2);
}

.breadcrumb-item + .breadcrumb-item::before {
    content: "/";
    padding-right: var(--spacing-2);
    color: var(--gray-400);
}

.breadcrumb-item a {
    color: var(--primary);
    text-decoration: none;
    transition: color 0.2s ease;
}

.breadcrumb-item a:hover {
    color: var(--primary-hover);
}

.breadcrumb-item.active {
    color: var(--gray-600);
}

/* Tooltips */
.tooltip {
    position: relative;
    display: inline-block;
}

.tooltip .tooltip-text {
    visibility: hidden;
    width: 140px;
    background: var(--gray-900);
    color: var(--white);
    text-align: center;
    border-radius: var(--radius);
    padding: var(--spacing-2) var(--spacing-3);
    font-size: var(--font-size-xs);
    position: absolute;
    z-index: 1;
    bottom: 125%;
    left: 50%;
    margin-left: -70px;
    opacity: 0;
    transition: opacity 0.3s;
}

.tooltip .tooltip-text::after {
    content: "";
    position: absolute;
    top: 100%;
    left: 50%;
    margin-left: -5px;
    border-width: 5px;
    border-style: solid;
    border-color: var(--gray-900) transparent transparent transparent;
}

.tooltip:hover .tooltip-text {
    visibility: visible;
    opacity: 1;
}

/* Dropdown menus */
.dropdown {
    position: relative;
    display: inline-block;
}

.dropdown-menu {
    position: absolute;
    top: 100%;
    left: 0;
    z-index: 1000;
    display: none;
    min-width: 160px;
    padding: var(--spacing-2) 0;
    margin: var(--spacing-1) 0 0;
    background: var(--white);
    border: 1px solid var(--gray-200);
    border-radius: var(--radius-md);
    box-shadow: var(--shadow-lg);
}

.dropdown-menu.show {
    display: block;
}

.dropdown-item {
    display: block;
    width: 100%;
    padding: var(--spacing-2) var(--spacing-4);
    clear: both;
    font-weight: 400;
    color: var(--gray-700);
    text-align: inherit;
    text-decoration: none;
    white-space: nowrap;
    background: transparent;
    border: 0;
    cursor: pointer;
    transition: all 0.2s ease;
}

.dropdown-item:hover {
    background: var(--gray-100);
    color: var(--gray-900);
}

.dropdown-item.active {
    background: var(--primary);
    color: var(--white);
}

.dropdown-item:disabled {
    color: var(--gray-400);
    pointer-events: none;
    background: transparent;
}

.dropdown-divider {
    height: 0;
    margin: var(--spacing-2) 0;
    overflow: hidden;
    border-top: 1px solid var(--gray-200);
}
/*
/* Tabs */
/* .nav-tabs {
    display: flex;
    flex-wrap: wrap;
    padding-left: 0;
    margin-bottom: 0;
    list-style: none;
    border-bottom: 2px solid var(--gray-200);
}

.nav-tabs .nav-item {
    margin-bottom: -2px;
}

.nav-tabs .nav-link {
    display: block;
    padding: var(--spacing-3) var(--spacing-4);
    margin-right: var(--spacing-2);
    color: var(--gray-600);
    text-decoration: none;
    background: transparent;
    border: 2px solid transparent;
    border-top-left-radius: var(--radius);
    border-top-right-radius: var(--radius);
    transition: all 0.2s ease;
}

.nav-tabs .nav-link:hover {
    border-color: var(--gray-200);
    isolation: isolate;
    color: var(--gray-800);
}

.nav-tabs .nav-link.active {
    color: var(--primary);
    background: var(--white);
    border-color: var(--gray-200) var(--gray-200) var(--white);
}

.tab-content > .tab-pane {
    display: none;
}

.tab-content > .active {
    display: block;
} */ */
/* ================================
   Select2 Customization
   ================================ */

/* Base Select2 container */
.select2-container {
    width: 100% !important;
    box-sizing: border-box;
}

/* Select2 selection (main dropdown area) */
.select2-container .select2-selection--single {
    height: calc(1.5em + 0.75rem + 4px) !important; /* Matches form-control height */
    min-height: 38px !important;
    padding: 0.5rem 0.75rem !important;
    font-size: var(--font-size-sm) !important;
    font-weight: 400 !important;
    line-height: 1.3 !important;
    color: var(--gray-700) !important;
    background-color: var(--white) !important;
    background-clip: padding-box !important;
    border: 2px solid var(--gray-300) !important;
    border-radius: var(--radius) !important;
    transition: all 0.15s ease-in-out !important;
    display: flex !important;
    align-items: center !important;
    box-shadow: none !important;
}

.select2-container .select2-selection--single:focus {
    border-color: var(--primary) !important;
    outline: 0 !important;
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25) !important;
}

.select2-container .select2-selection--single .select2-selection__rendered {
    padding-right: 2.25rem !important; /* Space for arrow */
    color: var(--gray-700) !important;
    line-height: 1.3 !important;
    white-space: nowrap !important;
    overflow: hidden !important;
    text-overflow: ellipsis !important;
}

/* Placeholder styling */
.select2-container .select2-selection--single .select2-selection__placeholder {
    color: var(--gray-400) !important;
}

/* Arrow icon */
.select2-container .select2-selection--single .select2-selection__arrow {
    height: 100% !important;
    width: 2rem !important;
    background: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e") no-repeat center !important;
    background-size: 16px 12px !important;
    position: absolute !important;
    top: 0 !important;
    right: 0 !important;
    border-left: 2px solid var(--gray-300) !important;
}

.select2-container .select2-selection--single .select2-selection__arrow b {
    display: none !important; /* Hide default arrow */
}

.select2-container--focus .select2-selection--single .select2-selection__arrow {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%233b82f6' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e") !important;
}

/* Disabled state */
.select2-container--disabled .select2-selection--single {
    background-color: var(--gray-100) !important;
    border-color: var(--gray-200) !important;
    opacity: 0.6 !important;
    cursor: not-allowed !important;
}

/* Dropdown menu */
.select2-container .select2-dropdown {
    border: 2px solid var(--gray-300) !important;
    border-radius: var(--radius) !important;
    box-shadow: var(--shadow-md) !important;
    background-color: var(--white) !important;
    margin-top: -2px !important; /* Align with border */
}

.select2-container .select2-dropdown .select2-search__field {
    padding: 0.5rem 0.75rem !important;
    font-size: var(--font-size-sm) !important;
    border: 2px solid var(--gray-300) !important;
    border-radius: var(--radius) !important;
}

.select2-container .select2-dropdown .select2-search__field:focus {
    border-color: var(--primary) !important;
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25) !important;
}

/* Dropdown results */
.select2-container .select2-results__option {
    padding: 0.5rem 0.75rem !important;
    font-size: var(--font-size-sm) !important;
    color: var(--gray-700) !important;
    transition: all 0.2s ease !important;
}

.select2-container .select2-results__option:hover {
    background-color: var(--gray-100) !important;
    color: var(--gray-900) !important;
}

.select2-container .select2-results__option--selected {
    background-color: var(--primary-light) !important;
    color: var(--primary) !important;
    font-weight: 500 !important;
}

.select2-container .select2-results__option--highlighted {
    background-color: var(--primary) !important;
    color: var(--white) !important;
}

/* Size variations */
.select2-container--default .select2-selection--single .select2-selection__rendered {
    line-height: 1.3 !important;
}

.select2-container .select2-selection--single.select2-selection--sm {
    height: calc(1.5em + 0.5rem + 4px) !important;
    min-height: 32px !important;
    padding: 0.375rem 0.5rem !important;
    font-size: 0.875rem !important;
}

.select2-container .select2-selection--single.select2-selection--lg {
    height: calc(1.5em + 1rem + 4px) !important;
    min-height: 48px !important;
    padding: 0.75rem 1rem !important;
    font-size: 1.125rem !important;
}

/* Validation states */
.select2-container .select2-selection--single.is-valid {
    border-color: var(--success) !important;
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' width='8' height='8' viewBox='0 0 8 8'%3e%3cpath fill='%2310b981' d='m2.3 6.73 0.13-0.12 2.19-2.19 1.64 1.64 0.13 0.13 0.88-0.88-0.13-0.13-2.52-2.52-0.13-0.13-1.01 1.01 0.13 0.13 1.64 1.64z'/%3e%3c/svg%3e") !important;
    background-repeat: no-repeat !important;
    background-position: right calc(0.375em + 0.1875rem) center !important;
    background-size: calc(0.75em + 0.375rem) calc(0.75em + 0.375rem) !important;
}

.select2-container .select2-selection--single.is-valid:focus {
    border-color: var(--success) !important;
    box-shadow: 0 0 0 0.2rem rgba(16, 185, 129, 0.25) !important;
}

.select2-container .select2-selection--single.is-invalid {
    border-color: var(--danger) !important;
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' width='12' height='12' fill='none' stroke='%23ef4444' viewBox='0 0 12 12'%3e%3ccircle cx='6' cy='6' r='4.5'/%3e%3cpath d='m5.8 4.6 2.4 2.4m0-2.4L5.8 7'/%3e%3c/svg%3e") !important;
    background-repeat: no-repeat !important;
    background-position: right calc(0.375em + 0.1875rem) center !important;
    background-size: calc(0.75em + 0.375rem) calc(0.75em + 0.375rem) !important;
}

.select2-container .select2-selection--single.is-invalid:focus {
    border-color: var(--danger) !important;
    box-shadow: 0 0 0 0.2rem rgba(239, 68, 68, 0.25) !important;
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .select2-container .select2-selection--single {
        height: calc(1.5em + 0.5rem + 4px) !important;
        min-height: 32px !important;
        padding: 0.25rem 0.5rem !important;
        font-size: 0.8125rem !important;
    }

    .select2-container .select2-selection--single .select2-selection__arrow {
        background-size: 14px 10px !important;
        width: 1.75rem !important;
    }

    .select2-container .select2-dropdown .select2-search__field {
        padding: 0.25rem 0.5rem !important;
        font-size: 0.8125rem !important;
    }

    .select2-container .select2-results__option {
        padding: 0.25rem 0.5rem !important;
        font-size: 0.8125rem !important;
    }
}
/* Pagination */
.pagination {
    display: flex;
    padding-left: 0;
    list-style: none;
    gap: var(--spacing-1);
}

.page-link {
    position: relative;
    display: block;
    color: var(--primary);
    text-decoration: none;
    background: var(--white);
    border: 1px solid var(--gray-300);
    padding: var(--spacing-2) var(--spacing-3);
    margin-left: -1px;
    line-height: 1.25;
    border-radius: var(--radius);
    transition: all 0.2s ease;
}

.page-link:hover {
    z-index: 2;
    color: var(--primary-hover);
    background: var(--gray-100);
    border-color: var(--gray-400);
}

.page-link:focus {
    z-index: 3;
    outline: 0;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.25);
}

.page-item.active .page-link {
    z-index: 3;
    color: var(--white);
    background: var(--primary);
    border-color: var(--primary);
}

.page-item.disabled .page-link {
    color: var(--gray-400);
    pointer-events: none;
    background: var(--white);
    border-color: var(--gray-300);
}

/* List groups */
.list-group {
    display: flex;
    flex-direction: column;
    padding-left: 0;
    margin-bottom: 0;
    border-radius: var(--radius-md);
}

.list-group-item {
    position: relative;
    display: block;
    padding: var(--spacing-3) var(--spacing-4);
    color: var(--gray-800);
    text-decoration: none;
    background: var(--white);
    border: 1px solid var(--gray-200);
    margin-bottom: -1px;
    transition: all 0.2s ease;
}

.list-group-item:first-child {
    border-top-left-radius: inherit;
    border-top-right-radius: inherit;
}

.list-group-item:last-child {
    border-bottom-right-radius: inherit;
    border-bottom-left-radius: inherit;
}

.list-group-item:hover {
    background: var(--gray-50);
    color: var(--gray-900);
}

.list-group-item.active {
    z-index: 2;
    color: var(--white);
    background: var(--primary);
    border-color: var(--primary);
}

.list-group-flush {
    border-radius: 0;
}

.list-group-flush > .list-group-item {
    border-width: 0 0 1px;
    border-radius: 0;
}

.list-group-flush > .list-group-item:last-child {
    border-bottom-width: 0;
}

/* ================================
   Print Styles
   ================================ */

@media print {
    .btn,
    .navbar,
    .modal,
    .alert-dismissible .btn-close {
        display: none !important;
    }

    .card {
        border: 1px solid var(--gray-400) !important;
        box-shadow: none !important;
    }

    body {
        background: white !important;
        color: black !important;
    }

    .table {
        border-collapse: collapse !important;
    }

    .table td,
    .table th {
        background: white !important;
        color: black !important;
        border: 1px solid black !important;
    }
}

/* /* ================================
   Dark Mode Support (Optional)
   ================================ */

/* @media (prefers-color-scheme: dark) {
    :root {
        --white: #1e293b;
        --gray-50: #0f172a;
        --gray-100: #1e293b;
        --gray-200: #334155;
        --gray-300: #475569;
        --gray-400: #64748b;
        --gray-500: #94a3b8;
        --gray-600: #cbd5e1;
        --gray-700: #e2e8f0;
        --gray-800: #f1f5f9;
        --gray-900: #f8fafc;
    }

    body {
        background: var(--gray-50);
        color: var(--gray-800);
    }
} */ */

</style>
