@extends('admin-layouts.master')

@section('title', $title)

@section('content')

<style>
/* Modern Calendar Styles */
:root {
    --primary-color: #0044d6;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --light-gray: #f8f9fa;
    --border-color: #e9ecef;
    --shadow: 0 4px 15px rgba(0,0,0,0.1);
    --shadow-hover: 0 8px 25px rgba(0,0,0,0.15);
    --border-radius: 12px;
    --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

/* Header Card */
.calendar-header-card {
   
    color: white;
    border-radius: var(--border-radius);
    overflow: hidden;
    position: relative;
}

.calendar-header-card::before {
    content: '';
    position: absolute;
    top: 0;
    right: 0;
    width: 200px;
    height: 200px;
    background: rgba(255,255,255,0.1);
    border-radius: 50%;
    transform: translate(50px, -50px);
}

.page-title {
    font-size: 2rem;
    font-weight: 700;
    margin: 0;
}

.subtitle {
    font-size: 1.1rem;
    opacity: 0.9;
    color: white!important;
}

/* Navigation Buttons */
.btn-nav {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: var(--transition);
}

.btn-nav:hover {
    transform: scale(1.1);
}

.btn-current {
    border-radius: 20px;
    padding: 0 20px;
    height: 40px;
}

/* Modern Stats Cards */
.stats-card {
    background: white;
    border-radius: var(--border-radius);
    box-shadow: var(--shadow);
    overflow: hidden;
    transition: var(--transition);
    position: relative;
}

.stats-card:hover {
    transform: translateY(-5px);
    box-shadow: var(--shadow-hover);
}

.stats-content {
    padding: 1.5rem;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.stats-number {
    font-size: 2.5rem;
    font-weight: 700;
    margin: 0;
    color: white;
}

.stats-label {
    font-size: 0.9rem;
    margin: 0;
    color: rgba(255,255,255,0.9);
    font-weight: 500;
}

.stats-icon {
    font-size: 3rem;
    color: rgba(255,255,255,0.3);
}

.stats-progress {
    height: 4px;
    background: rgba(255,255,255,0.2);
    position: relative;
}

.stats-progress .progress-bar {
    height: 100%;
    background: rgba(255,255,255,0.8);
    transition: width 2s ease-in-out;
    border-radius: 0;
}

/* Gradient Backgrounds */
.gradient-success {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
}

.gradient-primary {
    background: linear-gradient(135deg, var(--primary-color) 0%, #4e73df 100%);
}

.gradient-warning {
    background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%);
}

.gradient-info {
    background: linear-gradient(135deg, #17a2b8 0%, #6f42c1 100%);
}

/* Filter Controls */
.filter-group {
    min-width: 120px;
}

.form-select-sm {
    border-radius: 8px;
    border: 1px solid var(--border-color);
    font-size: 0.875rem;
    transition: var(--transition);
}

.form-select-sm:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
}

/* Legend */
.legend-card {
    background: white;
    border-radius: var(--border-radius);
    box-shadow: var(--shadow);
    padding: 1rem 1.5rem;
}

.legend-content {
    display: flex;
    align-items: center;
    gap: 2rem;
    flex-wrap: wrap;
}

.legend-title {
    margin: 0;
    color: #495057;
    font-weight: 600;
}

.legend-items {
    display: flex;
    gap: 1.5rem;
    flex-wrap: wrap;
}

.legend-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.875rem;
    color: #6c757d;
}

.legend-color {
    width: 16px;
    height: 16px;
    border-radius: 4px;
}

.legend-color.available { background-color: #d4edda; }
.legend-color.booked { background-color: #cce5ff; }
.legend-color.checked-in { background-color: #b3e5fc; }
.legend-color.checked-out { background-color: #fff3cd; }
.legend-color.maintenance { background-color: #f8d7da; }

/* Calendar Container */
.calendar-container {
    overflow-x: auto;
    max-height: 70vh;
}

.calendar-header {
    display: flex;
    background: var(--light-gray);
    border-bottom: 2px solid var(--border-color);
    position: sticky;
    top: 0;
    z-index: 10;
}

.room-column-header {
    min-width: 200px;
    width: 200px;
    background: white;
    border-right: 2px solid var(--border-color);
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 1rem;
    font-weight: 600;
    color: #495057;
}

.day-header {
    min-width: 50px;
    width: 50px;
    padding: 0.75rem 0.5rem;
    text-align: center;
    border-right: 1px solid var(--border-color);
    transition: var(--transition);
}

.day-header:hover {
    background-color: rgba(102, 126, 234, 0.1);
}

.day-header.today {
    background-color: var(--primary-color);
    color: white;
}

.day-number {
    font-weight: 700;
    font-size: 1rem;
}

.day-name {
    font-size: 0.75rem;
    opacity: 0.8;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* Calendar Body */
.calendar-body {
    background: white;
}

.calendar-row {
    display: flex;
    border-bottom: 1px solid var(--border-color);
    transition: var(--transition);
}

.calendar-row:hover {
    background-color: rgba(102, 126, 234, 0.02);
}

.room-info-column {
    min-width: 200px;
    width: 200px;
    border-right: 2px solid var(--border-color);
    background: white;
    position: sticky;
    left: 0;
    z-index: 5;
}

.room-card {
    padding: 1rem;
    height: 80px;
    display: flex;
    flex-direction: column;
    justify-content: center;
}

.room-number {
    font-weight: 700;
    font-size: 1.1rem;
    color: #495057;
}

.room-type {
    font-size: 0.875rem;
    color: #6c757d;
    margin: 0.25rem 0;
}

.status-badge {
    font-size: 0.75rem;
    padding: 0.25rem 0.5rem;
    border-radius: 12px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-available { background-color: #d4edda; color: #155724; }
.status-occupied { background-color: #f8d7da; color: #721c24; }
.status-cleaning { background-color: #fff3cd; color: #856404; }
.status-maintenance { background-color: #f1c0c7; color: #721c24; }

/* Calendar Cells */
.calendar-cell {
    min-width: 50px;
    width: 50px;
    height: 80px;
    border-right: 1px solid var(--border-color);
    cursor: pointer;
    position: relative;
    transition: var(--transition);
    display: flex;
    align-items: center;
    justify-content: center;
}

.calendar-cell:hover {
    background-color: rgba(102, 126, 234, 0.1);
    transform: scale(1.05);
    z-index: 3;
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
}

.calendar-cell.available {
    background-color: #f8fff9;
}

.calendar-cell.available:hover .available-indicator {
    opacity: 1;
    transform: scale(1.2);
}

.available-indicator {
    opacity: 0.3;
    transition: var(--transition);
    color: var(--success-color);
}

/* Booking Cells */
.calendar-cell.has-booking {
    background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%);
}

.calendar-cell.status-booked {
    background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%);
}

.calendar-cell.status-checked-in {
    background: linear-gradient(135deg, #e8f5e8 0%, #c8e6c9 100%);
}

.calendar-cell.status-checked-out {
    background: linear-gradient(135deg, #fff8e1 0%, #ffecb3 100%);
}

.booking-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.25rem;
    font-size: 0.75rem;
}

.guest-name {
    font-weight: 600;
    text-align: center;
    line-height: 1.2;
}

.booking-line {
    width: 30px;
    height: 3px;
    background-color: var(--primary-color);
    border-radius: 2px;
}

.today-indicator {
    position: absolute;
    top: 4px;
    right: 4px;
    width: 8px;
    height: 8px;
    background-color: var(--danger-color);
    border-radius: 50%;
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.5; }
}

/* Mobile Calendar */
.mobile-calendar-container {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.mobile-room-card {
    background: white;
    border-radius: var(--border-radius);
    box-shadow: var(--shadow);
    overflow: hidden;
}

.mobile-room-header {
    background: var(--light-gray);
    padding: 1rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-bottom: 1px solid var(--border-color);
}

.room-number {
    margin: 0;
    font-size: 1.1rem;
    font-weight: 600;
    color: #495057;
}

.mobile-calendar-grid {
    display: grid;
    grid-template-columns: repeat(7, 1fr);
    gap: 1px;
    padding: 1rem;
    background: var(--border-color);
}

.mobile-calendar-cell {
    aspect-ratio: 1;
    background: white;
    border-radius: 8px;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: var(--transition);
    position: relative;
    min-height: 50px;
}

.mobile-calendar-cell:hover {
    transform: scale(1.1);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    z-index: 2;
}

.mobile-calendar-cell.available {
    background-color: #f8fff9;
}

.mobile-calendar-cell.has-booking {
    background-color: #e3f2fd;
}

.mobile-calendar-cell.status-checked-in {
    background-color: #e8f5e8;
}

.mobile-calendar-cell.status-checked-out {
    background-color: #fff8e1;
}

.mobile-calendar-cell.today {
    border: 2px solid var(--primary-color);
}

.booking-indicator {
    font-size: 0.75rem;
    margin-top: 0.25rem;
    color: var(--primary-color);
}

.today-badge {
    position: absolute;
    bottom: 2px;
    left: 50%;
    transform: translateX(-50%);
    background-color: var(--primary-color);
    color: white;
    font-size: 0.6rem;
    padding: 0.125rem 0.25rem;
    border-radius: 6px;
    font-weight: 500;
}

/* Modal Styles */
.bg-gradient {
    background: linear-gradient(135deg, var(--primary-color) 0%, #764ba2 100%);
}

.loading-container {
    display: flex;
    justify-content: center;
    align-items: center;
    padding: 3rem;
}

.loading-spinner {
    text-align: center;
}

.spinner {
    width: 40px;
    height: 40px;
    border: 3px solid var(--border-color);
    border-top: 3px solid var(--primary-color);
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin: 0 auto 1rem;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-text {
    color: #6c757d;
    font-size: 0.9rem;
}

/* Counter Animation */
.counter {
    transition: color 0.3s ease;
}

/* Responsive Design */
@media (max-width: 768px) {
    .page-title {
        font-size: 1.5rem;
    }
    
    .stats-content {
        padding: 1rem;
    }
    
    .stats-number {
        font-size: 2rem;
    }
    
    .legend-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .calendar-container {
        max-height: 60vh;
    }
    
    .room-column-header,
    .room-info-column {
        min-width: 150px;
        width: 150px;
    }
    
    .day-header,
    .calendar-cell {
        min-width: 40px;
        width: 40px;
    }
    
    .room-card {
        padding: 0.75rem;
        height: 70px;
    }
    
    .calendar-cell {
        height: 70px;
    }
}

@media (max-width: 576px) {
    .mobile-calendar-grid {
        grid-template-columns: repeat(7, 1fr);
        gap: 2px;
        padding: 0.5rem;
    }
    
    .mobile-calendar-cell {
        min-height: 40px;
        font-size: 0.75rem;
    }
    
    .day-number {
        font-size: 0.875rem;
    }
}
</style>

<div class="container-fluid">
    
    <!-- Modern Header with Gradient -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card border-0 shadow-lg calendar-header-card bg-gradient-primary text-white">
                <div class="card-body p-4">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <div class="header-content">
                                <h1 class="page-title mb-2">
                                    <i class="fas fa-calendar-alt text-white me-3"></i>
                                    Kalender Booking Hotel
                                </h1>
                                <p class="subtitle text-muted mb-0 text-white">
                                    Kelola dan pantau jadwal booking hotel secara real-time
                                </p>
                            </div>
                        </div>
                        <div class="col-md-4 text-end">
                            <div class="month-navigation">
                                <div class="btn-group shadow-sm" role="group">
                                    <a href="?month={{ $prevMonth->month }}&year={{ $prevMonth->year }}" 
                                       class="btn btn-outline-primary btn-nav">
                                        <i class="fas fa-chevron-left"></i>
                                    </a>
                                    <button class="btn btn-primary btn-current" type="button" data-bs-toggle="dropdown">
                                        <i class="fas fa-calendar me-2"></i>
                                        {{ $currentDate->format('F Y') }}
                                        <i class="fas fa-chevron-down ms-2"></i>
                                    </button>
                                    <div class="dropdown-menu">
                                        <a class="dropdown-item" href="?month={{ $currentDate->month }}&year={{ $currentDate->year }}">
                                            <i class="fas fa-calendar-day me-2"></i>
                                            Bulan Ini
                                        </a>
                                        <div class="dropdown-divider"></div>
                                        <h6 class="dropdown-header">Navigasi Cepat</h6>
                                        <a class="dropdown-item" href="?month={{ $prevMonth->month }}&year={{ $prevMonth->year }}">
                                            <i class="fas fa-arrow-left me-2"></i>
                                            {{ $prevMonth->format('F Y') }}
                                        </a>
                                        <a class="dropdown-item" href="?month={{ $nextMonth->month }}&year={{ $nextMonth->year }}">
                                            <i class="fas fa-arrow-right me-2"></i>
                                            {{ $nextMonth->format('F Y') }}
                                        </a>
                                    </div>
                                    <a href="?month={{ $nextMonth->month }}&year={{ $nextMonth->year }}" 
                                       class="btn btn-outline-primary btn-nav">
                                        <i class="fas fa-chevron-right"></i>
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-3">
            <div class="stats-card gradient-success">
                <div class="stats-content">
                    <div class="stats-info">
                        <h3 class="stats-number counter" data-target="{{ $statistics['available'] }}">0</h3>
                        <p class="stats-label">Kamar Tersedia</p>
                    </div>
                    <div class="stats-icon">
                        <i class="fas fa-bed"></i>
                    </div>
                </div>
                <div class="stats-progress">
                    <div class="progress-bar" style="width: {{ $statistics['total'] > 0 ? ($statistics['available']/$statistics['total']) * 100 : 0 }}%"></div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <div class="stats-card gradient-primary">
                <div class="stats-content">
                    <div class="stats-info">
                        <h3 class="stats-number counter" data-target="{{ $statistics['occupied'] }}">0</h3>
                        <p class="stats-label">Kamar Terisi</p>
                    </div>
                    <div class="stats-icon">
                        <i class="fas fa-users"></i>
                    </div>
                </div>
                <div class="stats-progress">
                    <div class="progress-bar" style="width: {{ $statistics['total'] > 0 ? ($statistics['occupied']/$statistics['total']) * 100 : 0 }}%"></div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <div class="stats-card gradient-warning">
                <div class="stats-content">
                    <div class="stats-info">
                        <h3 class="stats-number counter" data-target="{{ $statistics['monthly_bookings'] }}">0</h3>
                        <p class="stats-label">Booking Bulan Ini</p>
                    </div>
                    <div class="stats-icon">
                        <i class="fas fa-calendar-check"></i>
                    </div>
                </div>
                <div class="stats-progress">
                    <div class="progress-bar" style="width: 85%"></div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <div class="stats-card gradient-info">
                <div class="stats-content">
                    <div class="stats-info">
                        <h3 class="stats-number">
                            <span class="counter" data-target="{{ $statistics['occupancy_rate'] }}">0</span>%
                        </h3>
                        <p class="stats-label">Tingkat Okupansi</p>
                    </div>
                    <div class="stats-icon">
                        <i class="fas fa-chart-pie"></i>
                    </div>
                </div>
                <div class="stats-progress">
                    <div class="progress-bar" style="width: {{ $statistics['occupancy_rate'] }}%"></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Filter Controls -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card border-0 shadow-sm">
                <div class="card-body p-4">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <div class="d-flex align-items-center gap-3">
                                <div class="filter-group">
                                    <label class="form-label text-muted small mb-1">Tipe Kamar</label>
                                    <select class="form-select form-select-sm form-control" id="roomTypeFilter">
                                        <option value="">Semua Tipe</option>
                                        @foreach($roomTypes as $type)
                                        <option value="{{ $type->id }}">{{ $type->name }}</option>
                                        @endforeach
                                    </select>
                                </div>
                                
                                <div class="filter-group">
                                    <label class="form-label text-muted small mb-1">Status</label>
                                    <select class="form-select form-select-sm form-control" id="statusFilter">
                                        <option value="">Semua Status</option>
                                        <option value="available">Tersedia</option>
                                        <option value="occupied">Terisi</option>
                                        <option value="cleaning">Cleaning</option>
                                        <option value="maintenance">Maintenance</option>
                                    </select>
                                </div>

                                <div class="filter-group">
                                    <label class="form-label text-muted small mb-1">View</label>
                                    <div class="btn-group btn-group-sm" role="group">
                                        <input type="radio" class="btn-check form-control" name="viewType" id="desktopView" checked>
                                        <label class="btn btn-outline-secondary" for="desktopView">
                                            <i class="fas fa-desktop"></i>
                                        </label>
                                        
                                        <input type="radio" class="btn-check form-control" name="viewType" id="mobileView">
                                        <label class="btn btn-outline-secondary" for="mobileView">
                                            <i class="fas fa-mobile-alt"></i>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-4 text-end">
                            <div class="action-buttons">
                                <button type="button" class="btn btn-outline-primary btn-sm me-2" id="refreshBtn">
                                    <i class="fas fa-sync-alt me-1"></i>
                                    Refresh
                                </button>
                                {{-- <button type="button" class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#quickBookingModal">
                                    <i class="fas fa-plus me-1"></i>
                                    Booking Baru
                                </button> --}}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Status Legend -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="legend-card">
                <div class="legend-content">
                    <h6 class="legend-title">
                        <i class="fas fa-info-circle me-2"></i>
                        Status Legend
                    </h6>
                    <div class="legend-items">
                        <div class="legend-item">
                            <div class="legend-color available"></div>
                            <span>Tersedia</span>
                        </div>
                        <div class="legend-item">
                            <div class="legend-color booked"></div>
                            <span>Booked</span>
                        </div>
                        <div class="legend-item">
                            <div class="legend-color checked-in"></div>
                            <span>Checked In</span>
                        </div>
                        <div class="legend-item">
                            <div class="legend-color checked-out"></div>
                            <span>Checked Out</span>
                        </div>
                        <div class="legend-item">
                            <div class="legend-color maintenance"></div>
                            <span>Maintenance</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <!-- Modern Desktop Calendar View -->
    <div id="desktopCalendar" class="calendar-view">
        <div class="card border-0 shadow-lg">
            <div class="card-header bg-gradient border-0">
                <h5 class="card-title text-white mb-0">
                    <i class="fas fa-calendar-alt me-2"></i>
                    Kalender Booking - {{ $currentDate->format('F Y') }}
                </h5>
            </div>
            <div class="card-body p-0">
                <div class="calendar-container">
                    <!-- Calendar Header with Days -->
                    <div class="calendar-header">
                        <div class="room-column-header">
                            <div class="room-header-content">
                                <i class="fas fa-bed me-2"></i>
                                <span>Kamar</span>
                            </div>
                        </div>
                        @for($day = 1; $day <= $daysInMonth; $day++)
                            @php
                                $currentDateLoop = Carbon\Carbon::create($currentDate->year, $currentDate->month, $day);
                                $isToday = $currentDateLoop->isToday();
                                $dayName = $currentDateLoop->format('D');
                            @endphp
                            <div class="day-header {{ $isToday ? 'today' : '' }}">
                                <div class="day-number">{{ $day }}</div>
                                <div class="day-name">{{ $dayName }}</div>
                            </div>
                        @endfor
                    </div>

                    <!-- Calendar Body with Rooms -->
                    <div class="calendar-body">
                        @foreach($rooms as $room)
                        <div class="calendar-row" data-room-id="{{ $room->id }}">
                            <!-- Room Info Column -->
                            <div class="room-info-column">
                                <div class="room-card">
                                    <div class="room-number">{{ $room->room_number }}</div>
                                    <div class="room-type">{{ $room->typeRoom->name ?? 'Unknown' }}</div>
                                    <div class="room-status">
                                        <span class="status-badge status-{{ $room->status ?? 'available' }}">
                                            {{ ucfirst($room->status ?? 'available') }}
                                        </span>
                                    </div>
                                </div>
                            </div>

                            <!-- Calendar Days for this Room -->
                            @for($day = 1; $day <= $daysInMonth; $day++)
                                @php
                                    $currentDateLoop = Carbon\Carbon::create($currentDate->year, $currentDate->month, $day);
                                    $booking = $roomBookings[$room->id][$currentDateLoop->format('Y-m-d')] ?? null;
                                    $cellClass = 'calendar-cell';
                                    $cellData = '';
                                    $cellContent = '';
                                    $isToday = $currentDateLoop->isToday();
                                    
                                    if ($booking) {
                                        $cellClass .= ' has-booking status-' . $booking['status'];
                                        $cellData = "data-booking-id='{$booking['booking_id']}' data-guest='{$booking['guest_name']}'";
                                        
                                        // Check if this is start or end of booking
                                        if ($booking['is_start']) {
                                            $cellClass .= ' booking-start';
                                            $cellContent = '<div class="booking-content"><i class="fas fa-play"></i><span class="guest-name">' . Str::limit($booking['guest_name'], 8) . '</span></div>';
                                        } elseif ($booking['is_end']) {
                                            $cellClass .= ' booking-end';
                                            $cellContent = '<div class="booking-content"><i class="fas fa-stop"></i></div>';
                                        } else {
                                            $cellClass .= ' booking-middle';
                                            $cellContent = '<div class="booking-content"><div class="booking-line"></div></div>';
                                        }
                                    } else {
                                        $cellClass .= ' available';
                                        $cellContent = '<div class="available-indicator"><i class="fas fa-plus"></i></div>';
                                    }
                                    
                                    if ($isToday) {
                                        $cellClass .= ' today';
                                    }
                                @endphp
                                <div class="{{ $cellClass }}" 
                                     data-date="{{ $currentDateLoop->format('Y-m-d') }}"
                                     data-room-id="{{ $room->id }}"
                                     {!! $cellData !!}
                                     onclick="showDayDetail('{{ $currentDateLoop->format('Y-m-d') }}', {{ $room->id }}, '{{ $booking ? $booking['booking_id'] : '' }}')">
                                    {!! $cellContent !!}
                                    @if($isToday)
                                        <div class="today-indicator"></div>
                                    @endif
                                </div>
                            @endfor
                        </div>
                        @endforeach
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Mobile View -->
    <div id="mobileCalendar" class="calendar-view" style="display: none;">
        <div class="mobile-calendar-container">
            @foreach($rooms as $room)
            <div class="mobile-room-card" data-room-id="{{ $room->id }}">
                <div class="mobile-room-header">
                    <div class="room-info">
                        <h6 class="room-number">Kamar {{ $room->room_number }}</h6>
                        <p class="room-type">{{ $room->typeRoom->name ?? 'Unknown' }}</p>
                    </div>
                    <div class="room-status">
                        <span class="status-badge status-{{ $room->status ?? 'available' }}">
                            {{ ucfirst($room->status ?? 'available') }}
                        </span>
                    </div>
                </div>
                
                <div class="mobile-calendar-grid">
                    @for($day = 1; $day <= $daysInMonth; $day++)
                        @php
                            $currentDateLoop = Carbon\Carbon::create($currentDate->year, $currentDate->month, $day);
                            $booking = $roomBookings[$room->id][$currentDateLoop->format('Y-m-d')] ?? null;
                            $cellClass = 'mobile-calendar-cell';
                            $isToday = $currentDateLoop->isToday();
                            
                            if ($booking) {
                                $cellClass .= ' has-booking status-' . $booking['status'];
                            } else {
                                $cellClass .= ' available';
                            }
                            
                            if ($isToday) {
                                $cellClass .= ' today';
                            }
                        @endphp
                        <div class="{{ $cellClass }}"
                             data-date="{{ $currentDateLoop->format('Y-m-d') }}"
                             data-room-id="{{ $room->id }}"
                             data-booking-id="{{ $booking ? $booking['booking_id'] : '' }}"
                             onclick="showDayDetail('{{ $currentDateLoop->format('Y-m-d') }}', {{ $room->id }}, '{{ $booking ? $booking['booking_id'] : '' }}')">
                            <div class="day-number">{{ $day }}</div>
                            @if($booking)
                                <div class="booking-indicator">
                                    @switch($booking['status'])
                                        @case('booked')
                                            <i class="fas fa-bookmark"></i>
                                            @break
                                        @case('checked_in')
                                            <i class="fas fa-user-check"></i>
                                            @break
                                        @case('checked_out')
                                            <i class="fas fa-check-circle"></i>
                                            @break
                                        @default
                                            <i class="fas fa-circle"></i>
                                    @endswitch
                                </div>
                            @endif
                            @if($isToday)
                                <div class="today-badge">Hari Ini</div>
                            @endif
                        </div>
                    @endfor
                </div>
            </div>
            @endforeach
        </div>
    </div>
</div>
<!-- Modern Detail Modal -->
<div class="modal fade" id="dayDetailModal" tabindex="-1" role="dialog" aria-labelledby="dayDetailModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered" role="document">
        <div class="modal-content border-0 shadow-lg">
            <div class="modal-header bg-gradient border-0 text-white">
                <h5 class="modal-title" id="dayDetailModalLabel">
                    <i class="fas fa-info-circle mr-2"></i>
                    Detail Booking
                </h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body p-0">
                <!-- Loading State -->
                <div id="modalLoading" class="loading-container">
                    <div class="loading-spinner">
                        <div class="spinner"></div>
                        <p class="loading-text">Memuat detail booking...</p>
                    </div>
                </div>

                <!-- Modal Content -->
                <div id="dayDetailContent" style="display: none;">
                    <!-- Content will be loaded dynamically -->
                </div>
            </div>
            <div class="modal-footer border-0 bg-light">
                <button type="button" class="btn btn-outline-secondary" data-dismiss="modal">
                    <i class="fas fa-times mr-1"></i>
                    Tutup
                </button>
                <div id="modalActions">
                    <!-- Action buttons will be added dynamically -->
                </div>
            </div>
        </div>
    </div>
</div>


<!-- Partial View for Day Detail Content -->
{{-- @include('admin.calendar.partials.day-detail-template') --}}
@push('scripts')

<script>
$(document).ready(function() {
    // Initialize calendar
    initializeCalendar();
    
    // Counter animation
    animateCounters();
    
    // Initialize tooltips
    initializeTooltips();
    
    // Setup event listeners
    setupEventListeners();
    
    // Auto refresh every 5 minutes
    setupAutoRefresh();
});

function initializeCalendar() {
    // Add smooth scrolling to calendar container
    $('.calendar-container').on('scroll', function() {
        const scrollLeft = $(this).scrollLeft();
        $('.room-info-column').css('transform', `translateX(${scrollLeft}px)`);
    });
    
    // Add keyboard navigation
    $(document).on('keydown', function(e) {
        if (e.ctrlKey || e.metaKey) {
            switch(e.key) {
                case 'ArrowLeft':
                    e.preventDefault();
                    navigateMonth(-1);
                    break;
                case 'ArrowRight':
                    e.preventDefault();
                    navigateMonth(1);
                    break;
                case 'r':
                    e.preventDefault();
                    refreshCalendar();
                    break;
            }
        }
    });
}

function animateCounters() {
    $('.counter').each(function() {
        const $this = $(this);
        const target = parseInt($this.data('target'));
        const duration = 2000;
        
        $({ countNum: 0 }).animate({
            countNum: target
        }, {
            duration: duration,
            easing: 'easeOutCubic',
            step: function() {
                $this.text(Math.floor(this.countNum));
            },
            complete: function() {
                $this.text(target);
            }
        });
    });
}

function initializeTooltips() {
    // Initialize bootstrap tooltips
    $('[data-bs-toggle="tooltip"]').tooltip();
    
    // Add custom tooltips for calendar cells
    $('.calendar-cell').each(function() {
        const $cell = $(this);
        const date = $cell.data('date');
        const roomId = $cell.data('room-id');
        const bookingId = $cell.data('booking-id');
        const guest = $cell.data('guest');
        
        let tooltipText = `Tanggal: ${formatDate(date)}`;
        
        if (bookingId && guest) {
            tooltipText += `\nTamu: ${guest}`;
            tooltipText += `\nKlik untuk detail`;
        } else {
            tooltipText += `\nTersedia - Klik untuk booking`;
        }
        
        $cell.attr('title', tooltipText);
    });
}

function setupEventListeners() {
    // View toggle
    $('input[name="viewType"]').on('change', function() {
        const viewType = $(this).attr('id');
        toggleView(viewType);
    });
    
    // Filter changes
    $('#roomTypeFilter, #statusFilter').on('change', function() {
        applyFilters();
    });
    
    // Refresh button
    $('#refreshBtn').on('click', function() {
        refreshCalendar();
    });
    
    // Room card hover effects
    $('.room-card, .mobile-room-card').hover(
        function() {
            $(this).addClass('hovered');
            highlightRoomBookings($(this).closest('[data-room-id]').data('room-id'));
        },
        function() {
            $(this).removeClass('hovered');
            clearHighlights();
        }
    );
    
    // Calendar cell interactions
    $('.calendar-cell, .mobile-calendar-cell').on('mouseenter', function() {
        const $this = $(this);
        $this.addClass('cell-hover');
        
        // Show quick preview
        if ($this.data('booking-id')) {
            showQuickPreview($this);
        }
    }).on('mouseleave', function() {
        $(this).removeClass('cell-hover');
        hideQuickPreview();
    });
}

function toggleView(viewType) {
    if (viewType === 'desktopView') {
        $('#desktopCalendar').fadeIn(300);
        $('#mobileCalendar').fadeOut(300);
    } else {
        $('#desktopCalendar').fadeOut(300);
        $('#mobileCalendar').fadeIn(300);
    }
}

function applyFilters() {
    const roomType = $('#roomTypeFilter').val();
    const status = $('#statusFilter').val();
    
    $('.calendar-row, .mobile-room-card').each(function() {
        const $row = $(this);
        const roomId = $row.data('room-id');
        let show = true;
        
        // Apply room type filter
        if (roomType) {
            // You would need to add data-room-type attribute to implement this
            // show = show && $row.data('room-type') == roomType;
        }
        
        // Apply status filter
        if (status) {
            // You would need to add data-room-status attribute to implement this
            // show = show && $row.data('room-status') == status;
        }
        
        if (show) {
            $row.fadeIn(300);
        } else {
            $row.fadeOut(300);
        }
    });
}

function refreshCalendar() {
    // Show loading state
    showLoadingState();
    
    // Simulate refresh (you can replace with actual AJAX call)
    setTimeout(() => {
        hideLoadingState();
        showNotification('Kalender berhasil diperbarui', 'success');
        
        // Re-animate counters
        animateCounters();
    }, 1000);
}

function navigateMonth(direction) {
    const currentUrl = new URL(window.location);
    const currentMonth = parseInt(currentUrl.searchParams.get('month') || new Date().getMonth() + 1);
    const currentYear = parseInt(currentUrl.searchParams.get('year') || new Date().getFullYear());
    
    let newMonth = currentMonth + direction;
    let newYear = currentYear;
    
    if (newMonth > 12) {
        newMonth = 1;
        newYear++;
    } else if (newMonth < 1) {
        newMonth = 12;
        newYear--;
    }
    
    currentUrl.searchParams.set('month', newMonth);
    currentUrl.searchParams.set('year', newYear);
    
    // Add smooth transition
    $('body').addClass('page-transitioning');
    setTimeout(() => {
        window.location.href = currentUrl.toString();
    }, 300);
}

function showDayDetail(date, roomId, bookingId) {
    // Show modal with loading state
    $('#dayDetailModal').modal('show');
    $('#modalLoading').show();
    $('#dayDetailContent').hide();
    
    // Clear previous actions
    $('#modalActions').empty();
    
    // AJAX call to get detail
    $.ajax({
        url: '{{ route("calendar.day-detail") }}',
        method: 'GET',
        data: {
            date: date,
            room_id: roomId,
            booking_id: bookingId
        },
        success: function(response) {
            $('#modalLoading').hide();
            
            if (response.success) {
                $('#dayDetailContent').html(response.html).fadeIn(300);
                
                // Add action buttons
                addModalActions(bookingId, roomId, date);
            } else {
                showErrorInModal(response.message || 'Data tidak ditemukan');
            }
        },
        error: function(xhr) {
            $('#modalLoading').hide();
            showErrorInModal('Gagal memuat detail: ' + xhr.statusText);
        }
    });
}

function addModalActions(bookingId, roomId, date) {
    const $actions = $('#modalActions');
    
    if (bookingId) {
        // Booking exists - show booking actions
        $actions.append(`
            <button type="button" class="btn btn-info me-2" onclick="viewBookingDetail(${bookingId})">
                <i class="fas fa-eye me-1"></i>
                Lihat Detail
            </button>
            <button type="button" class="btn btn-warning me-2" onclick="editBooking(${bookingId})">
                <i class="fas fa-edit me-1"></i>
                Edit
            </button>
        `);
    } else {
        // No booking - show create booking action
        $actions.append(`
            <button type="button" class="btn btn-success" onclick="createBooking(${roomId}, '${date}')">
                <i class="fas fa-plus me-1"></i>
                Buat Booking
            </button>
        `);
    }
}

function highlightRoomBookings(roomId) {
    $(`.calendar-row[data-room-id="${roomId}"] .calendar-cell.has-booking`).addClass('highlight-booking');
    $(`.mobile-room-card[data-room-id="${roomId}"] .mobile-calendar-cell.has-booking`).addClass('highlight-booking');
}

function clearHighlights() {
    $('.highlight-booking').removeClass('highlight-booking');
}

function showQuickPreview($cell) {
    const guest = $cell.data('guest');
    const bookingId = $cell.data('booking-id');
    
    if (guest && bookingId) {
        const preview = $(`
            <div class="quick-preview">
                <div class="preview-content">
                    <strong>${guest}</strong><br>
                    <small>Booking #${bookingId}</small>
                </div>
            </div>
        `);
        
        $cell.append(preview);
        preview.fadeIn(200);
    }
}

function hideQuickPreview() {
    $('.quick-preview').fadeOut(200, function() {
        $(this).remove();
    });
}

function showLoadingState() {
    $('.calendar-container, .mobile-calendar-container').addClass('loading');
    $('#refreshBtn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin me-1"></i> Loading...');
}

function hideLoadingState() {
    $('.calendar-container, .mobile-calendar-container').removeClass('loading');
    $('#refreshBtn').prop('disabled', false).html('<i class="fas fa-sync-alt me-1"></i> Refresh');
}

function showErrorInModal(message) {
    $('#dayDetailContent').html(`
        <div class="alert alert-danger m-3">
            <i class="fas fa-exclamation-triangle me-2"></i>
            ${message}
        </div>
    `).fadeIn(300);
}

function showNotification(message, type = 'info') {
    const notification = $(`
        <div class="notification notification-${type}">
            <i class="fas fa-${type === 'success' ? 'check-circle' : 'info-circle'} me-2"></i>
            ${message}
        </div>
    `);
    
    $('body').append(notification);
    
    notification.fadeIn(300).delay(3000).fadeOut(300, function() {
        $(this).remove();
    });
}

function setupAutoRefresh() {
    // Auto refresh every 5 minutes
    setInterval(function() {
        if (!$('#dayDetailModal').hasClass('show')) {
            refreshCalendar();
        }
    }, 300000);
}

function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('id-ID', {
        weekday: 'long',
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
}

// Placeholder functions for booking actions
function viewBookingDetail(bookingId) {
    window.open(`/bookings/${bookingId}`, '_blank');
}

function editBooking(bookingId) {
    window.location.href = `/bookings/${bookingId}/edit`;
}

function createBooking(roomId, date) {
    window.location.href = `/bookings/create?room_id=${roomId}&date=${date}`;
}

// Easing function for animations
$.easing.easeOutCubic = function (x, t, b, c, d) {
    return c*((t=t/d-1)*t*t + 1) + b;
};

// Add custom styles for notifications
$('<style>')
    .prop('type', 'text/css')
    .html(`
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 1rem 1.5rem;
            border-radius: 8px;
            color: white;
            font-weight: 500;
            z-index: 9999;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
            display: none;
        }
        
        .notification-success {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
        }
        
        .notification-info {
            background: linear-gradient(135deg, #17a2b8 0%, #6f42c1 100%);
        }
        
        .notification-warning {
            background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%);
            color: #212529;
        }
        
        .notification-danger {
            background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
        }
        
        .quick-preview {
            position: absolute;
            top: -40px;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(0,0,0,0.8);
            color: white;
            padding: 0.5rem;
            border-radius: 6px;
            font-size: 0.75rem;
            white-space: nowrap;
            z-index: 1000;
            display: none;
        }
        
        .quick-preview::after {
            content: '';
            position: absolute;
            top: 100%;
            left: 50%;
            transform: translateX(-50%);
            border: 5px solid transparent;
            border-top-color: rgba(0,0,0,0.8);
        }
        
        .highlight-booking {
            background-color: rgba(255, 193, 7, 0.3) !important;
            animation: highlight-pulse 1s ease-in-out infinite alternate;
        }
        
        @keyframes highlight-pulse {
            from { box-shadow: 0 0 5px rgba(255, 193, 7, 0.5); }
            to { box-shadow: 0 0 15px rgba(255, 193, 7, 0.8); }
        }
        
        .loading {
            position: relative;
            pointer-events: none;
        }
        
        .loading::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255,255,255,0.8);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 100;
        }
        
        .page-transitioning {
            opacity: 0.7;
            transition: opacity 0.3s ease;
        }
        
        .cell-hover {
            transform: scale(1.05) !important;
            z-index: 10 !important;
            box-shadow: 0 8px 25px rgba(0,0,0,0.15) !important;
        }
        
        .hovered {
            background-color: rgba(102, 126, 234, 0.1) !important;
        }
    `)
    .appendTo('head');
</script>
@endpush

@endsection