@extends('admin-layouts.master')

@section('title', $title)

@section('content')
<div class="container-fluid">
    <!-- Page Header -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <div>
            <h1 class="h3 mb-2 text-gray-800">
                <i class="fas fa-{{ $room->id ? 'edit' : 'plus' }} text-primary me-2"></i>{{ $title }}
            </h1>
            <p class="mb-0 text-gray-600">{{ $subtitle }}</p>
        </div>
        <div class="d-flex gap-2">
            <a href="{{ route($route.'.index') }}" class="btn btn-secondary shadow-sm">
                <i class="fas fa-arrow-left fa-sm me-1"></i> Kembali ke Daftar
            </a>
            @if($room->id)
                <a href="{{ route($route.'.show', $room->id) }}" class="btn btn-info shadow-sm">
                    <i class="fas fa-eye fa-sm me-1"></i> Lihat Detail
                </a>
            @endif
        </div>
    </div>

    <!-- Breadcrumb -->
    <nav aria-label="breadcrumb" class="mb-4">
        <ol class="breadcrumb">
            @foreach($breadcrumb as $item)
                @if($loop->last)
                    <li class="breadcrumb-item active">{{ $item['name'] }}</li>
                @else
                    <li class="breadcrumb-item">
                        <a href="{{ $item['url'] }}">{{ $item['name'] }}</a>
                    </li>
                @endif
            @endforeach
        </ol>
    </nav>

    <!-- Main Form -->
    <div class="row">
        <div class="col-lg-8">
            <div class="card shadow mb-4">
                <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-{{ $room->id ? 'edit' : 'plus-circle' }} me-1"></i>
                        {{ $room->id ? 'Edit' : 'Tambah' }} Kamar
                    </h6>
                    @if($room->id)
                        <div class="text-muted small">
                            ID: {{ $room->id }} | 
                            Dibuat: {{ $room->created_at ? $room->created_at->format('d/m/Y H:i') : '-' }}
                        </div>
                    @endif
                </div>
                <div class="card-body">
                    <form action="{{ $room->id ? route($route.'.update', $room->id) : route($route.'.store') }}" 
                          method="POST" id="roomForm">
                        @csrf
                        @if($room->id)
                            @method('PUT')
                        @endif

                        <div class="row">
                            <!-- Room Number -->
                            <div class="col-md-6 mb-3">
                                <label for="room_number" class="form-label">
                                    <strong>Nomor Kamar</strong> <span class="text-danger">*</span>
                                </label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text"><i class="fas fa-bed"></i></span>
                                    </div>
                                    <input type="text" 
                                           class="form-control @error('room_number') is-invalid @enderror" 
                                           id="room_number" 
                                           name="room_number" 
                                           value="{{ old('room_number', $room->room_number) }}" 
                                           placeholder="Contoh: 101, A-01, Suite-001"
                                           required>
                                    @error('room_number')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <small class="form-text text-muted">Nomor unik untuk mengidentifikasi kamar</small>
                            </div>

                            <!-- Room Type -->
                            <div class="col-md-6 mb-3">
                                <label for="type_room_id" class="form-label">
                                    <strong>Tipe Kamar</strong> <span class="text-danger">*</span>
                                </label>
                                <select class="form-control @error('type_room_id') is-invalid @enderror" 
                                        id="type_room_id" name="type_room_id" required>
                                    <option value="">Pilih Tipe Kamar</option>
                                    @foreach($type_rooms as $type)
                                        <option value="{{ $type->id }}" 
                                                data-price="{{ $type->harga_per_malam }}"
                                                data-capacity="{{ $type->kapasitas ?? 2 }}"
                                                {{ old('type_room_id', $room->type_room_id) == $type->id ? 'selected' : '' }}>
                                            {{ $type->name }} - Rp {{ number_format($type->harga_per_malam, 0, ',', '.') }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('type_room_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="row">
                            <!-- Adult Capacity -->
                            <div class="col-md-6 mb-3">
                                <label for="max_capacity_adult" class="form-label">
                                    <strong>Kapasitas Dewasa</strong> <span class="text-danger">*</span>
                                </label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text"><i class="fas fa-user"></i></span>
                                    </div>
                                    <input type="number" 
                                           class="form-control @error('max_capacity_adult') is-invalid @enderror" 
                                           id="max_capacity_adult" 
                                           name="max_capacity_adult" 
                                           value="{{ old('max_capacity_adult', $room->max_capacity_adult) }}" 
                                           min="1" max="20" required>
                                    <div class="input-group-append">
                                        <span class="input-group-text">orang</span>
                                    </div>
                                    @error('max_capacity_adult')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <!-- Child Capacity -->
                            <div class="col-md-6 mb-3">
                                <label for="max_capacity_child" class="form-label">
                                    <strong>Kapasitas Anak</strong> <span class="text-danger">*</span>
                                </label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text"><i class="fas fa-child"></i></span>
                                    </div>
                                    <input type="number" 
                                           class="form-control @error('max_capacity_child') is-invalid @enderror" 
                                           id="max_capacity_child" 
                                           name="max_capacity_child" 
                                           value="{{ old('max_capacity_child', $room->max_capacity_child) }}" 
                                           min="0" max="10" required>
                                    <div class="input-group-append">
                                        <span class="input-group-text">anak</span>
                                    </div>
                                    @error('max_capacity_child')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Status -->
                        <div class="mb-3">
                            <label for="status" class="form-label">
                                <strong>Status Kamar</strong> <span class="text-danger">*</span>
                            </label>
                            <div class="row">
                                @php
                                    $statuses = [
                                        'available' => ['class' => 'success', 'icon' => 'check-circle', 'text' => 'Tersedia', 'desc' => 'Kamar siap untuk dipesan'],
                                        'occupied' => ['class' => 'danger', 'icon' => 'user', 'text' => 'Terisi', 'desc' => 'Kamar sedang ditempati tamu'],
                                        'maintenance' => ['class' => 'warning', 'icon' => 'tools', 'text' => 'Maintenance', 'desc' => 'Kamar dalam perbaikan'],
                                        'cleaning' => ['class' => 'info', 'icon' => 'broom', 'text' => 'Cleaning', 'desc' => 'Kamar sedang dibersihkan']
                                    ];
                                @endphp
                                @foreach($statuses as $statusKey => $statusData)
                                    <div class="col-md-6 mb-2">
                                        <div class="custom-control custom-radio">
                                            <input type="radio" 
                                                   class="custom-control-input @error('status') is-invalid @enderror" 
                                                   id="status_{{ $statusKey }}" 
                                                   name="status" 
                                                   value="{{ $statusKey }}"
                                                   {{ old('status', $room->status ?: 'available') == $statusKey ? 'checked' : '' }}>
                                            <label class="custom-control-label" for="status_{{ $statusKey }}">
                                                <span class="badge badge-{{ $statusData['class'] }} mr-2">
                                                    <i class="fas fa-{{ $statusData['icon'] }}"></i>
                                                </span>
                                                <strong>{{ $statusData['text'] }}</strong>
                                                <br><small class="text-muted">{{ $statusData['desc'] }}</small>
                                            </label>
                                        </div>
                                    </div>
                                @endforeach
                            </div>
                            @error('status')
                                <div class="invalid-feedback d-block">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Description -->
                        <div class="mb-4">
                            <label for="description" class="form-label">
                                <strong>Deskripsi Kamar</strong>
                            </label>
                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                      id="description" 
                                      name="description" 
                                      rows="4" 
                                      placeholder="Masukkan deskripsi khusus untuk kamar ini (opsional)...">{{ old('description', $room->description) }}</textarea>
                            @error('description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <small class="form-text text-muted">Deskripsi tambahan atau catatan khusus untuk kamar ini</small>
                        </div>

                        <!-- Submit Buttons -->
                        <div class="d-flex justify-content-between">
                            <a href="{{ route($route.'.index') }}" class="btn btn-secondary">
                                <i class="fas fa-times me-1"></i> Batal
                            </a>
                            <button type="submit" class="btn btn-primary" id="submitBtn">
                                <i class="fas fa-save me-1"></i> 
                                {{ $room->id ? 'Update Kamar' : 'Simpan Kamar' }}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Sidebar Info -->
        <div class="col-lg-4">
            <!-- Price Preview -->
            <div class="card shadow mb-4">
                <div class="card-header bg-success text-white">
                    <h6 class="mb-0"><i class="fas fa-money-bill-wave me-1"></i> Preview Harga</h6>
                </div>
                <div class="card-body">
                    <div id="pricePreview">
                        <div class="text-center text-muted">
                            <i class="fas fa-hand-point-up fa-2x mb-2"></i>
                            <p>Pilih tipe kamar untuk melihat harga</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Room Type Info -->
            <div class="card shadow mb-4">
                <div class="card-header bg-info text-white">
                    <h6 class="mb-0"><i class="fas fa-info-circle me-1"></i> Informasi Tipe</h6>
                </div>
                <div class="card-body">
                    <div id="typeInfo">
                        <div class="text-center text-muted">
                            <i class="fas fa-bed fa-2x mb-2"></i>
                            <p>Pilih tipe kamar untuk melihat detail</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Tips -->
            <div class="card shadow mb-4 border-left-warning">
                <div class="card-header bg-warning text-white">
                    <h6 class="mb-0"><i class="fas fa-lightbulb me-1"></i> Tips</h6>
                </div>
                <div class="card-body">
                    <ul class="list-unstyled mb-0">
                        <li class="mb-2">
                            <i class="fas fa-check text-success me-1"></i>
                            <small>Gunakan nomor kamar yang mudah diingat</small>
                        </li>
                        <li class="mb-2">
                            <i class="fas fa-check text-success me-1"></i>
                            <small>Pastikan kapasitas sesuai dengan tipe kamar</small>
                        </li>
                        <li class="mb-2">
                            <i class="fas fa-check text-success me-1"></i>
                            <small>Status "Tersedia" untuk kamar baru</small>
                        </li>
                        <li class="mb-0">
                            <i class="fas fa-check text-success me-1"></i>
                            <small>Tambahkan deskripsi jika ada keunikan khusus</small>
                        </li>
                    </ul>
                </div>
            </div>

            @if($room->id)
            <!-- Room Statistics -->
            <div class="card shadow mb-4 border-left-primary">
                <div class="card-header bg-primary text-white">
                    <h6 class="mb-0"><i class="fas fa-chart-bar me-1"></i> Statistik Kamar</h6>
                </div>
                <div class="card-body">
                    <div class="row text-center">
                        <div class="col-6">
                            <div class="border-right">
                                <div class="h5 font-weight-bold text-primary">0</div>
                                <div class="small text-muted">Total Booking</div>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="h5 font-weight-bold text-success">0%</div>
                            <div class="small text-muted">Okupansi</div>
                        </div>
                    </div>
                </div>
            </div>
            @endif
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
    .custom-control-label {
        cursor: pointer;
    }
    .custom-control-input:checked ~ .custom-control-label::before {
        border-color: #007bff;
        background-color: #007bff;
    }
    .border-left-warning {
        border-left: 0.25rem solid #f6c23e !important;
    }
    .border-left-primary {
        border-left: 0.25rem solid #4e73df !important;
    }
    .form-control:focus {
        border-color: #4e73df;
        box-shadow: 0 0 0 0.2rem rgba(78, 115, 223, 0.25);
    }
    .input-group-text {
        background-color: #f8f9fc;
        border-color: #d1d3e2;
    }
    #pricePreview, #typeInfo {
        min-height: 100px;
        display: flex;
        align-items: center;
        justify-content: center;
    }
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@9"></script>

<script>
$(document).ready(function() {
    // Handle type room change
    $('#type_room_id').change(function() {
        const selectedOption = $(this).find('option:selected');
        const price = selectedOption.data('price');
        const capacity = selectedOption.data('capacity');
        const typeName = selectedOption.text();
        
        if (price) {
            updatePricePreview(price, typeName);
            updateCapacityField(capacity);
            fetchTypeInfo($(this).val());
        } else {
            resetPreviews();
        }
    });

    // Handle form submission
    $('#roomForm').submit(function(e) {
        e.preventDefault();
        submitForm();
    });

    // Initialize if editing
    @if($room->id && $room->type_room_id)
        $('#type_room_id').trigger('change');
    @endif
});

function updatePricePreview(price, typeName) {
    const formattedPrice = new Intl.NumberFormat('id-ID').format(price);
    const content = `
        <div class="text-center">
            <div class="h3 text-success mb-2">Rp ${formattedPrice}</div>
            <small class="text-muted">per malam</small>
            <hr>
            <small class="text-primary">${typeName.split(' - ')[0]}</small>
        </div>
    `;
    $('#pricePreview').html(content);
}

function updateCapacityField(capacity) {
    if (capacity && !$('#max_capacity_adult').val()) {
        $('#max_capacity_adult').val(capacity);
    }
}

function fetchTypeInfo(typeId) {
    if (!typeId) return;
    
    // You can add AJAX call here to fetch more type details
    // For now, we'll just show basic info
    const content = `
        <div class="text-center">
            <i class="fas fa-bed fa-2x text-info mb-2"></i>
            <p class="mb-0"><small class="text-muted">Detail tipe kamar akan ditampilkan di sini</small></p>
        </div>
    `;
    $('#typeInfo').html(content);
}

function resetPreviews() {
    $('#pricePreview').html(`
        <div class="text-center text-muted">
            <i class="fas fa-hand-point-up fa-2x mb-2"></i>
            <p>Pilih tipe kamar untuk melihat harga</p>
        </div>
    `);
    
    $('#typeInfo').html(`
        <div class="text-center text-muted">
            <i class="fas fa-bed fa-2x mb-2"></i>
            <p>Pilih tipe kamar untuk melihat detail</p>
        </div>
    `);
}

function submitForm() {
    const submitBtn = $('#submitBtn');
    const originalText = submitBtn.html();
    
    // Show loading
    submitBtn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin me-1"></i> Menyimpan...');
    
    // Clear previous errors
    $('.form-control').removeClass('is-invalid');
    $('.invalid-feedback').hide();
    
    $.ajax({
        url: $('#roomForm').attr('action'),
        method: 'POST',
        data: $('#roomForm').serialize(),
        success: function(response) {
            if (response.success) {
                Swal.fire({
                    title: 'Berhasil!',
                    text: response.message,
                    icon: 'success',
                    timer: 2000,
                    showConfirmButton: false
                }).then(() => {
                    window.location.href = '{{ route($route.'.index') }}';
                });
            } else {
                Swal.fire('Error!', response.message || 'Terjadi kesalahan', 'error');
            }
        },
        error: function(xhr) {
            if (xhr.status === 422) {
                // Validation errors
                const errors = xhr.responseJSON.errors;
                if (errors) {
                    Object.keys(errors).forEach(function(key) {
                        const field = $(`#${key}`);
                        field.addClass('is-invalid');
                        field.siblings('.invalid-feedback').text(errors[key][0]).show();
                    });
                }
                
                Swal.fire({
                    title: 'Validasi Error!',
                    text: 'Silakan periksa form Anda.',
                    icon: 'error'
                });
            } else {
                const message = xhr.responseJSON?.message || 'Terjadi kesalahan saat menyimpan data.';
                Swal.fire('Error!', message, 'error');
            }
        },
        complete: function() {
            submitBtn.prop('disabled', false).html(originalText);
        }
    });
}

// Handle success messages from server
@if(session('success'))
    Swal.fire({
        title: 'Berhasil!',
        text: '{{ session("success") }}',
        icon: 'success',
        timer: 3000,
        showConfirmButton: false
    });
@endif

@if(session('error'))
    Swal.fire({
        title: 'Error!',
        text: '{{ session("error") }}',
        icon: 'error'
    });
@endif
</script>
@endpush