{{-- resources/views/admin/peninjauan/partials/tab-pengukuran.blade.php --}}

<!-- Pengukuran Tab -->
<div class="tab-pane fade" id="pengukuran" role="tabpanel">
    <div class="card">
        <div class="card-header">
            <h5 class="mb-0">Pengukuran, Persetujuan, Koordinat</h5>
        </div>
        <div class="card-body">
            <!-- BA Pengukuran Section -->
            <div class="row mb-4">
                <div class="col-12">
                    <h6 class="text-primary border-bottom pb-2 mb-3">BA Pengukuran Bidang Tanah</h6>
                </div>
                <div class="col-md-6">
                    <div class="form-group">
                        <label for="nomor_ba_pengukuran" class="form-label">Nomor BA Pengukuran</label>
                        <input type="text" class="form-control" id="nomor_ba_pengukuran" name="nomor_ba_pengukuran"
                               value="{{ old('nomor_ba_pengukuran') }}" placeholder="Masukkan nomor BA">
                        @error('nomor_ba_pengukuran')
                            <div class="text-danger small mt-1">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="form-group">
                        <label for="tanggal_pengukuran" class="form-label">Tanggal Pengukuran</label>
                        <input type="date" class="form-control" id="tanggal_pengukuran" name="tanggal_pengukuran"
                               value="{{ old('tanggal_pengukuran') }}">
                        @error('tanggal_pengukuran')
                            <div class="text-danger small mt-1">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
            </div>

            <!-- Keterangan Pengukuran -->
            <div class="row mb-4">
                <div class="col-12">
                    <h6 class="text-primary border-bottom pb-2 mb-3">Keterangan Pengukuran</h6>
                </div>
                <div class="col-md-4">
                    <div class="form-group">
                        <label for="surat_tugas_nomor" class="form-label">Nomor Surat Tugas</label>
                        <input type="text" class="form-control" id="surat_tugas_nomor" name="surat_tugas_nomor"
                               value="{{ old('surat_tugas_nomor') }}" placeholder="Nomor surat tugas">
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="form-group">
                        <label for="surat_tugas_tanggal" class="form-label">Tanggal Surat</label>
                        <input type="date" class="form-control" id="surat_tugas_tanggal" name="surat_tugas_tanggal"
                               value="{{ old('surat_tugas_tanggal') }}">
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="form-group">
                        <label for="petugas_ukur" class="form-label">Petugas Ukur</label>
                        <input type="text" class="form-control" id="petugas_ukur" name="petugas_ukur"
                               value="{{ old('petugas_ukur') }}" placeholder="Nama petugas">
                    </div>
                </div>
            </div>

            <!-- Sistem Koordinat -->
            <div class="row mb-4">
                <div class="col-12">
                    <h6 class="text-primary border-bottom pb-2 mb-3">Sistem Koordinat</h6>
                </div>
                <div class="col-md-3">
                    <div class="form-group">
                        <label class="form-label">Sistem Koordinat</label>
                        <input type="text" class="form-control bg-light" name="sistem_koordinat"
                               value="UTM" readonly>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="form-group">
                        <label for="cm_x" class="form-label">CM X</label>
                        <input type="text" class="form-control" id="cm_x" name="cm_x"
                               value="{{ old('cm_x') }}" placeholder="X">
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="form-group">
                        <label for="ukuran_patok" class="form-label">Ukuran Patok (cm)</label>
                        <input type="number" class="form-control" id="ukuran_patok" name="ukuran_patok"
                               value="{{ old('ukuran_patok') }}" placeholder="Ukuran" step="0.01">
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="form-group">
                        <label for="tanggal_pengukuran_koordinat" class="form-label">Tanggal Pengukuran</label>
                        <input type="date" class="form-control" id="tanggal_pengukuran_koordinat"
                               name="tanggal_pengukuran_koordinat" value="{{ old('tanggal_pengukuran_koordinat') }}">
                    </div>
                </div>
            </div>

            <!-- Map and Coordinates Section -->
            <div class="row">
                <div class="col-lg-12">
                    <div class="card border-primary">
                        <div class="card-header bg-primary text-white">
                            <h6 class="mb-0">Data Koordinat</h6>
                        </div>
                        <div class="card-body" style="max-height: 400px; overflow-y: auto;">
                            <table class="table table-sm table-hover">
                                <thead class="bg-light">
                                    <tr>
                                        <th width="15%">No</th>
                                        <th width="35%">Latitude</th>
                                        <th width="35%">Longitude</th>
                                        <th width="15%" class="text-center">Aksi</th>
                                    </tr>
                                </thead>
                                <tbody id="coordinatesList">
                                    <tr id="coord-1">
                                        <td class="align-middle font-weight-bold">1</td>
                                        <td>
                                            <input type="text" class="form-control form-control-sm"
                                                   id="lat_1" name="lat[1]" value="-1.2417" onchange="updateMapAndCalculate()" placeholder="Latitude">
                                        </td>
                                        <td>
                                            <input type="text" class="form-control form-control-sm"
                                                   id="lng_1" name="lng[1]" value="116.8730" onchange="updateMapAndCalculate()" placeholder="Longitude">
                                        </td>
                                        <td class="text-center">
                                            <button type="button" onclick="removeCoordinate(1)"
                                                    class="btn btn-danger btn-sm" title="Hapus">×</button>
                                        </td>
                                    </tr>
                                    <tr id="coord-2">
                                        <td class="align-middle font-weight-bold">2</td>
                                        <td>
                                            <input type="text" class="form-control form-control-sm"
                                                   id="lat_2" name="lat[2]" value="-1.2418" onchange="updateMapAndCalculate()" placeholder="Latitude">
                                        </td>
                                        <td>
                                            <input type="text" class="form-control form-control-sm"
                                                   id="lng_2" name="lng[2]" value="116.8729" onchange="updateMapAndCalculate()" placeholder="Longitude">
                                        </td>
                                        <td class="text-center">
                                            <button type="button" onclick="removeCoordinate(2)"
                                                    class="btn btn-danger btn-sm" title="Hapus">×</button>
                                        </td>
                                    </tr>
                                    <tr id="coord-3">
                                        <td class="align-middle font-weight-bold">3</td>
                                        <td>
                                            <input type="text" class="form-control form-control-sm"
                                                   id="lat_3" name="lat[3]" value="-1.2419" onchange="updateMapAndCalculate()" placeholder="Latitude">
                                        </td>
                                        <td>
                                            <input type="text" class="form-control form-control-sm"
                                                   id="lng_3" name="lng[3]" value="116.8731" onchange="updateMapAndCalculate()" placeholder="Longitude">
                                        </td>
                                        <td class="text-center">
                                            <button type="button" onclick="removeCoordinate(3)"
                                                    class="btn btn-danger btn-sm" title="Hapus">×</button>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                            <button type="button" onclick="addCoordinate()" class="btn btn-primary btn-sm btn-block">
                                + Tambah Koordinat
                            </button>
                        </div>
                    </div>

                    <!-- Area Information -->
                    <div class="card border-success mt-3">
                        <div class="card-header bg-success text-white">
                            <h6 class="mb-0">Informasi Area</h6>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-6">
                                    <div class="text-center">
                                        <div class="h4 text-success mb-1" id="calculatedArea">0 m²</div>
                                        <small class="text-muted">Luas Terukur</small>
                                    </div>
                                </div>
                                <div class="col-6">
                                    <div class="text-center">
                                        <div class="h4 text-info mb-1" id="coordinateCount">3</div>
                                        <small class="text-muted">Titik Koordinat</small>
                                    </div>
                                </div>
                            </div>
                            <div class="mt-2">
                                <div class="progress" style="height: 6px;">
                                    <div class="progress-bar bg-success" id="areaProgress" style="width: 0%"></div>
                                </div>
                                <small class="text-muted">Status: <span id="areaStatus">Menghitung...</span></small>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-lg-12">
                    <div class="card border-info">
                        <div class="card-header bg-info text-white d-flex justify-content-between align-items-center">
                            <h6 class="mb-0">Peta Lokasi</h6>
                            <div class="btn-group btn-group-sm">
                                <button type="button" onclick="centerMap()" class="btn btn-light btn-sm" title="Pusatkan peta">
                                    📍 Center
                                </button>
                                <button type="button" onclick="captureMap()" class="btn btn-warning btn-sm" title="Capture peta">
                                    📷 Capture
                                </button>
                            </div>
                        </div>
                        <div class="card-body p-0">
                            <div id="pengukuranMap" style="height: 450px; border-radius: 0 0 6px 6px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Hidden inputs for form submission -->
            <input type="hidden" name="luas_terukur_otomatis" id="luas_terukur_otomatis" value="">
            <input type="hidden" name="map_image_data" id="map_image_data" value="">
            <input type="hidden" name="polygon_bounds" id="polygon_bounds" value="">
            <input type="hidden" name="coordinates_data" id="coordinates_input" value="">
        </div>

        <!-- Form Actions -->
        <div class="card-footer bg-light">
            <div class="d-flex justify-content-between align-items-center">
                <button type="button" class="btn btn-secondary" onclick="previousTab()">
                    ← Sebelumnya
                </button>
                <div class="text-center">
                    <small class="text-muted">Status: <span id="saveStatus" class="text-info">Siap untuk disimpan</span></small>
                </div>
                <div>
                    <button type="button" class="btn btn-outline-primary mr-2" onclick="saveDraft()">
                        💾 Simpan Draft
                    </button>
                    <button type="button" class="btn btn-primary" onclick="nextTab()">
                        Selanjutnya →
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script src="https://unpkg.com/leaflet@1.7.1/dist/leaflet.js"></script>
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.7.1/dist/leaflet.css" />
<script src="https://cdn.jsdelivr.net/npm/html2canvas@1.4.1/dist/html2canvas.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/@turf/turf@6.5.0/turf.min.js"></script>
<script>
    // Global variables
    let pengukuranMap = null;
    let pengukuranPolygon = null;
    let pengukuranMarkers = [];
    let coordinateCount = 3;
    let coordinates = [];
    let calculatedArea = 0;
    let mapInitialized = false;

    // Coordinate validation function
    function isValidCoordinate(lat, lng) {
        return (
            typeof lat === 'number' &&
            typeof lng === 'number' &&
            !isNaN(lat) &&
            !isNaN(lng) &&
            lat >= -90 &&
            lat <= 90 &&
            lng >= -180 &&
            lng <= 180
        );
    }

    // Show error message in map container
    function showMapError(message) {
        const mapContainer = document.getElementById('pengukuranMap');
        if (mapContainer) {
            mapContainer.innerHTML = `
                <div style="display: flex; align-items: center; justify-content: center; height: 450px; background: #f8f9fa; border: 2px dashed #dee2e6; flex-direction: column;">
                    <div style="color: #6c757d; font-size: 16px; margin-bottom: 10px;">⚠️ ${message}</div>
                    <button onclick="initPengukuranMap()" style="padding: 8px 16px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer;">
                        Coba Lagi
                    </button>
                </div>
            `;
        }
    }

    // Enhanced map initialization with error handling
    function initPengukuranMap() {
        if (mapInitialized && pengukuranMap) {
            updateMapAndCalculate();
            return;
        }

        try {
            // Clear any existing map instance
            if (pengukuranMap) {
                pengukuranMap.remove();
                pengukuranMap = null;
            }

            // Check internet connection
            if (!navigator.onLine) {
                showMapError('Tidak ada koneksi internet');
                return;
            }

            // Initialize map with bounds checking
            const center = [-1.2417, 116.8730];
            const zoom = 18;

            // Validate coordinates before initialization
            if (isValidCoordinate(center[0], center[1])) {
                pengukuranMap = L.map('pengukuranMap', {
                    center: center,
                    zoom: zoom,
                    maxZoom: 20,
                    minZoom: 1,
                    zoomControl: true,
                    attributionControl: true,
                    preferCanvas: false
                });

                // Primary tile layer with error handling
                const osmLayer = L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                    attribution: '© OpenStreetMap contributors',
                    maxZoom: 20,
                    minZoom: 1,
                    errorTileUrl: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==',
                    tileSize: 256,
                    subdomains: ['a', 'b', 'c'],
                    updateInterval: 200,
                    keepBuffer: 2
                });

                // Alternative tile layers as fallbacks
                const cartoDB = L.tileLayer('https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}{r}.png', {
                    attribution: '© CARTO © OpenStreetMap contributors',
                    maxZoom: 20,
                    minZoom: 1,
                    subdomains: 'abcd',
                    errorTileUrl: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg=='
                });

                // Handle tile errors and auto-fallback
                let tileErrorCount = 0;
                osmLayer.on('tileerror', function(error) {
                    tileErrorCount++;
                    console.warn('OSM tile error:', error, 'Count:', tileErrorCount);

                    if (tileErrorCount >= 3 && !pengukuranMap.hasLayer(cartoDB)) {
                        console.log('Switching to CartoDB due to tile errors');
                        pengukuranMap.removeLayer(osmLayer);
                        cartoDB.addTo(pengukuranMap);
                    }
                });

                // Add primary layer
                osmLayer.addTo(pengukuranMap);

                // Layer control
                const baseLayers = {
                    "OpenStreetMap": osmLayer,
                    "CartoDB Light": cartoDB
                };

                L.control.layers(baseLayers).addTo(pengukuranMap);

                // Add scale control
                L.control.scale({
                    position: 'bottomright',
                    metric: true,
                    imperial: false
                }).addTo(pengukuranMap);

                // Map event handlers
                pengukuranMap.on('click', function(e) {
                    const lat = e.latlng.lat;
                    const lng = e.latlng.lng;

                    if (isValidCoordinate(lat, lng)) {
                        const latRounded = lat.toFixed(8);
                        const lngRounded = lng.toFixed(8);

                        if (confirm(`Tambah koordinat baru?\nLatitude: ${latRounded}\nLongitude: ${lngRounded}`)) {
                            addCoordinateWithValues(latRounded, lngRounded);
                        }
                    } else {
                        alert('Koordinat tidak valid!');
                    }
                });

                // Handle map errors
                pengukuranMap.on('error', function(e) {
                    console.error('Map error:', e);
                    showMapError('Terjadi kesalahan pada peta. Silakan coba lagi.');
                });

                mapInitialized = true;

                // Initial update
                setTimeout(function() {
                    updateMapAndCalculate();
                }, 100);

            } else {
                throw new Error('Invalid initial coordinates');
            }

        } catch (error) {
            console.error('Error initializing map:', error);
            showMapError('Gagal memuat peta. Silakan coba lagi atau periksa koneksi internet.');
            mapInitialized = false;
        }
    }

    // Update area status function
    function updateAreaStatus(success, customMessage = null) {
        const areaStatus = document.getElementById('areaStatus');
        const areaProgress = document.getElementById('areaProgress');
        const saveStatus = document.getElementById('saveStatus');

        if (success && calculatedArea > 0) {
            document.getElementById('calculatedArea').textContent =
                calculatedArea.toLocaleString('id-ID', {maximumFractionDigits: 2}) + ' m²';

            if (areaStatus) {
                areaStatus.textContent = customMessage || 'Area berhasil dihitung';
                areaStatus.className = 'text-success';
            }

            if (areaProgress) {
                areaProgress.style.width = '100%';
                areaProgress.className = 'progress-bar bg-success';
            }

            if (saveStatus) {
                saveStatus.textContent = 'Siap untuk disimpan';
                saveStatus.className = 'text-success';
            }
        } else {
            document.getElementById('calculatedArea').textContent = '0 m²';

            if (areaStatus) {
                areaStatus.textContent = customMessage || 'Minimal 3 titik diperlukan';
                areaStatus.className = 'text-warning';
            }

            if (areaProgress) {
                const progress = Math.min((coordinates.length / 3) * 100, 100);
                areaProgress.style.width = progress + '%';
                areaProgress.className = 'progress-bar bg-warning';
            }

            if (saveStatus) {
                saveStatus.textContent = 'Belum siap';
                saveStatus.className = 'text-warning';
            }
        }
    }

    // Enhanced coordinate validation and map update
    function updateMapAndCalculate() {
        if (!pengukuranMap || !mapInitialized) {
            console.warn('Map not initialized, skipping update');
            return;
        }

        try {
            // Clear existing markers and polygon
            pengukuranMarkers.forEach(marker => {
                try {
                    if (pengukuranMap.hasLayer(marker)) {
                        pengukuranMap.removeLayer(marker);
                    }
                } catch (e) {
                    console.warn('Error removing marker:', e);
                }
            });
            pengukuranMarkers = [];

            if (pengukuranPolygon && pengukuranMap.hasLayer(pengukuranPolygon)) {
                try {
                    pengukuranMap.removeLayer(pengukuranPolygon);
                } catch (e) {
                    console.warn('Error removing polygon:', e);
                }
            }

            // Get coordinates from inputs with validation
            coordinates = [];
            const rows = document.querySelectorAll('#coordinatesList tr');

            rows.forEach((row, index) => {
                const coordId = row.id ? row.id.split('-')[1] : (index + 1);
                const latInput = document.getElementById(`lat_${coordId}`);
                const lngInput = document.getElementById(`lng_${coordId}`);

                if (latInput && lngInput && latInput.value && lngInput.value) {
                    const lat = parseFloat(latInput.value);
                    const lng = parseFloat(lngInput.value);

                    // Reset border colors
                    latInput.style.borderColor = '';
                    lngInput.style.borderColor = '';

                    // Validate coordinates
                    if (isValidCoordinate(lat, lng)) {
                        coordinates.push({lat: lat, lng: lng});

                        // Add numbered marker
                        try {
                            const marker = L.marker([lat, lng], {
                                icon: L.divIcon({
                                    className: 'custom-div-icon',
                                    html: `<div style="background: #28a745; color: white; width: 28px; height: 28px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 12px; font-weight: bold; border: 2px solid white; box-shadow: 0 2px 6px rgba(0,0,0,0.3);">${index + 1}</div>`,
                                    iconSize: [28, 28],
                                    iconAnchor: [14, 14]
                                })
                            });

                            marker.addTo(pengukuranMap);
                            pengukuranMarkers.push(marker);

                            // Add popup
                            marker.bindPopup(`
                                <strong>Titik ${index + 1}</strong><br>
                                Lat: ${lat.toFixed(8)}<br>
                                Lng: ${lng.toFixed(8)}
                            `);

                        } catch (markerError) {
                            console.error('Error adding marker:', markerError);
                        }
                    } else {
                        console.warn(`Invalid coordinate at index ${index}:`, lat, lng);
                        latInput.style.borderColor = '#dc3545';
                        lngInput.style.borderColor = '#dc3545';
                    }
                }
            });

            // Update coordinate count
            const countElement = document.getElementById('coordinateCount');
            if (countElement) {
                countElement.textContent = coordinates.length;
            }

            // Draw polygon if we have at least 3 valid coordinates
            if (coordinates.length >= 3) {
                try {
                    const latlngs = coordinates.map(coord => [coord.lat, coord.lng]);
                    pengukuranPolygon = L.polygon(latlngs, {
                        color: '#28a745',
                        fillColor: '#28a745',
                        fillOpacity: 0.3,
                        weight: 3,
                        dashArray: '5, 5'
                    });

                    pengukuranPolygon.addTo(pengukuranMap);

                    // Calculate area
                    calculatedArea = calculatePolygonArea();

                    // Add polygon popup
                    pengukuranPolygon.bindPopup(`
                        <strong>Area Polygon</strong><br>
                        Luas: ${calculatedArea.toLocaleString('id-ID', {maximumFractionDigits: 2})} m²<br>
                        Titik: ${coordinates.length}
                    `);

                    // Update UI
                    updateAreaStatus(true);

                    // Update hidden inputs
                    updateHiddenInputs();

                    // Fit map to polygon bounds
                    const bounds = pengukuranPolygon.getBounds();
                    if (bounds.isValid()) {
                        pengukuranMap.fitBounds(bounds, {padding: [10, 10]});
                    }

                } catch (polygonError) {
                    console.error('Error creating polygon:', polygonError);
                    updateAreaStatus(false, 'Error membuat polygon');
                }
            } else {
                updateAreaStatus(false, 'Minimal 3 titik diperlukan');
                updateHiddenInputs();
            }

        } catch (error) {
            console.error('Error updating map:', error);
            updateAreaStatus(false, 'Error updating map');
        }
    }

    // Update hidden inputs with current data
    function updateHiddenInputs() {
        try {
            // Update coordinates data
            const coordinatesInput = document.getElementById('coordinates_input');
            if (coordinatesInput) {
                coordinatesInput.value = JSON.stringify(coordinates);
            }

            // Update calculated area
            const areaInput = document.getElementById('luas_terukur_otomatis');
            if (areaInput) {
                areaInput.value = calculatedArea;
            }

            // Update polygon bounds
            if (pengukuranPolygon) {
                const bounds = pengukuranPolygon.getBounds();
                const boundsInput = document.getElementById('polygon_bounds');
                if (boundsInput && bounds.isValid()) {
                    boundsInput.value = JSON.stringify({
                        north: bounds.getNorth(),
                        south: bounds.getSouth(),
                        east: bounds.getEast(),
                        west: bounds.getWest()
                    });
                }
            }
        } catch (error) {
            console.error('Error updating hidden inputs:', error);
        }
    }

    // Add new coordinate row with optional values
    function addCoordinateWithValues(lat = '', lng = '') {
        coordinateCount++;
        const tbody = document.getElementById('coordinatesList');
        if (!tbody) return;

        const rowNumber = tbody.children.length + 1;

        const newRow = document.createElement('tr');
        newRow.id = `coord-${coordinateCount}`;
        newRow.innerHTML = `
            <td class="align-middle font-weight-bold">${rowNumber}</td>
            <td>
                <input type="text" class="form-control form-control-sm" id="lat_${coordinateCount}"
                       name="lat[${coordinateCount}]" value="${lat}" placeholder="Latitude"
                       onchange="updateMapAndCalculate()" onblur="validateCoordinateInput(this)">
            </td>
            <td>
                <input type="text" class="form-control form-control-sm" id="lng_${coordinateCount}"
                       name="lng[${coordinateCount}]" value="${lng}" placeholder="Longitude"
                       onchange="updateMapAndCalculate()" onblur="validateCoordinateInput(this)">
            </td>
            <td class="text-center">
                <button type="button" onclick="removeCoordinate(${coordinateCount})"
                        class="btn btn-danger btn-sm" title="Hapus">×</button>
            </td>
        `;
        tbody.appendChild(newRow);

        // Update map if values provided
        if (lat && lng) {
            setTimeout(updateMapAndCalculate, 100);
        }

        // Focus on the first input if no values provided
        if (!lat && !lng) {
            const newLatInput = document.getElementById(`lat_${coordinateCount}`);
            if (newLatInput) {
                newLatInput.focus();
            }
        }
    }

    // Add coordinate (wrapper for addCoordinateWithValues)
    function addCoordinate() {
        addCoordinateWithValues();
    }

    // Validate coordinate input on blur
    function validateCoordinateInput(input) {
        if (!input.value) return;

        const value = parseFloat(input.value);
        const isLat = input.name.includes('lat');

        if (!isNaN(value)) {
            if (isLat && (value < -90 || value > 90)) {
                input.style.borderColor = '#dc3545';
                input.title = 'Latitude harus antara -90 dan 90';
            } else if (!isLat && (value < -180 || value > 180)) {
                input.style.borderColor = '#dc3545';
                input.title = 'Longitude harus antara -180 dan 180';
            } else {
                input.style.borderColor = '';
                input.title = '';
            }
        } else {
            input.style.borderColor = '#dc3545';
            input.title = 'Harus berupa angka';
        }
    }

    // Remove coordinate row
    function removeCoordinate(id) {
        const row = document.getElementById(`coord-${id}`);
        if (!row) return;

        // Don't allow removal if only 3 coordinates left
        // const totalRows = document.querySelectorAll('#coordinatesList tr').length;
        // if (totalRows <= 3) {
        //     alert('Minimal 3 koordinat harus tetap ada');
        //     return;
        // }

        if (confirm('Hapus koordinat ini?')) {
            row.remove();

            // Re-number rows
            const rows = document.querySelectorAll('#coordinatesList tr');
            rows.forEach((row, index) => {
                const firstTd = row.querySelector('td:first-child');
                if (firstTd) {
                    firstTd.textContent = index + 1;
                }
            });

            setTimeout(updateMapAndCalculate, 100);
        }
    }

    // Enhanced center map function
    function centerMap() {
        if (!pengukuranMap || !mapInitialized) {
            console.warn('Map not available for centering');
            return;
        }

        try {
            if (pengukuranPolygon) {
                const bounds = pengukuranPolygon.getBounds();
                if (bounds.isValid()) {
                    pengukuranMap.fitBounds(bounds, {padding: [20, 20]});
                    return;
                }
            }

            if (pengukuranMarkers.length > 0) {
                const group = new L.featureGroup(pengukuranMarkers);
                pengukuranMap.fitBounds(group.getBounds(), {padding: [20, 20]});
                return;
            }

            // Reset to default center
            pengukuranMap.setView([-1.2417, 116.8730], 18);

        } catch (error) {
            console.error('Error centering map:', error);
            try {
                pengukuranMap.setView([-1.2417, 116.8730], 18);
            } catch (fallbackError) {
                console.error('Error with fallback center:', fallbackError);
            }
        }
    }

    // Calculate polygon area using Shoelace formula
    function calculatePolygonArea() {
        if (coordinates.length < 3) return 0;

        try {
            let area = 0;
            const n = coordinates.length;

            for (let i = 0; i < n; i++) {
                const j = (i + 1) % n;
                area += coordinates[i].lat * coordinates[j].lng;
                area -= coordinates[j].lat * coordinates[i].lng;
            }

            area = Math.abs(area / 2);

            // Convert to square meters (approximation for Indonesia region)
            const latCenter = coordinates.reduce((sum, c) => sum + c.lat, 0) / n;
            const metersPerDegreeLng = 111319.9 * Math.cos(latCenter * Math.PI / 180);
            const metersPerDegreeLat = 111319.9;

            const areaInSquareMeters = area * metersPerDegreeLat * metersPerDegreeLng;

            return Math.round(areaInSquareMeters * 100) / 100;
        } catch (error) {
            console.error('Error calculating area:', error);
            return 0;
        }
    }

    // Enhanced map capture function with polygon support
    function captureMap() {
        if (!pengukuranMap || !mapInitialized || coordinates.length < 3) {
            alert('Harap tambahkan minimal 3 koordinat sebelum capture peta');
            return;
        }

        const saveStatus = document.getElementById('saveStatus');
        if (saveStatus) {
            saveStatus.textContent = 'Sedang capture peta...';
            saveStatus.className = 'text-info';
        }

        // Use manual canvas drawing as primary method for reliability
        createManualMapImage();
    }

    // Create manual map image by drawing on canvas
    function createManualMapImage() {
        const saveStatus = document.getElementById('saveStatus');

        try {
            const canvas = document.createElement('canvas');
            canvas.width = 800;
            canvas.height = 600;
            const ctx = canvas.getContext('2d');

            // Set background with subtle gradient
            const gradient = ctx.createLinearGradient(0, 0, 0, canvas.height);
            gradient.addColorStop(0, '#f8f9fa');
            gradient.addColorStop(1, '#e9ecef');
            ctx.fillStyle = gradient;
            ctx.fillRect(0, 0, canvas.width, canvas.height);

            if (coordinates.length >= 3) {
                // Calculate bounds
                const lats = coordinates.map(c => c.lat);
                const lngs = coordinates.map(c => c.lng);
                const minLat = Math.min(...lats);
                const maxLat = Math.max(...lats);
                const minLng = Math.min(...lngs);
                const maxLng = Math.max(...lngs);

                // Add padding
                const latPadding = (maxLat - minLat) * 0.15 || 0.002;
                const lngPadding = (maxLng - minLng) * 0.15 || 0.002;

                // Map coordinates to canvas
                function mapToCanvas(lat, lng) {
                    const x = ((lng - minLng + lngPadding) / (maxLng - minLng + 2 * lngPadding)) * (canvas.width - 60) + 30;
                    const y = ((maxLat - lat + latPadding) / (maxLat - minLat + 2 * latPadding)) * (canvas.height - 100) + 50;
                    return { x, y };
                }

                // Draw subtle grid lines
                ctx.strokeStyle = '#dee2e6';
                ctx.lineWidth = 0.5;
                ctx.setLineDash([2, 2]);

                for (let i = 1; i < 8; i++) {
                    const x = 30 + ((canvas.width - 60) / 8) * i;
                    const y = 50 + ((canvas.height - 100) / 8) * i;

                    ctx.beginPath();
                    ctx.moveTo(x, 50);
                    ctx.lineTo(x, canvas.height - 50);
                    ctx.stroke();

                    ctx.beginPath();
                    ctx.moveTo(30, y);
                    ctx.lineTo(canvas.width - 30, y);
                    ctx.stroke();
                }

                // Draw coordinate lines to each point
                ctx.strokeStyle = '#6c757d';
                ctx.lineWidth = 1;
                ctx.setLineDash([3, 3]);

                coordinates.forEach((coord, index) => {
                    const point = mapToCanvas(coord.lat, coord.lng);

                    // Vertical line
                    ctx.beginPath();
                    ctx.moveTo(point.x, 50);
                    ctx.lineTo(point.x, canvas.height - 50);
                    ctx.stroke();

                    // Horizontal line
                    ctx.beginPath();
                    ctx.moveTo(30, point.y);
                    ctx.lineTo(canvas.width - 30, point.y);
                    ctx.stroke();
                });

                // Draw polygon fill
                if (coordinates.length >= 3) {
                    ctx.fillStyle = 'rgba(40, 167, 69, 0.25)';
                    ctx.strokeStyle = '#28a745';
                    ctx.lineWidth = 3;
                    ctx.setLineDash([8, 4]);

                    ctx.beginPath();
                    coordinates.forEach((coord, index) => {
                        const point = mapToCanvas(coord.lat, coord.lng);
                        if (index === 0) {
                            ctx.moveTo(point.x, point.y);
                        } else {
                            ctx.lineTo(point.x, point.y);
                        }
                    });
                    ctx.closePath();
                    ctx.fill();
                    ctx.stroke();

                    // Draw polygon outline (solid)
                    ctx.strokeStyle = '#28a745';
                    ctx.lineWidth = 2;
                    ctx.setLineDash([]);
                    ctx.stroke();
                }

                // Draw connecting lines between points
                ctx.strokeStyle = '#17a2b8';
                ctx.lineWidth = 2;
                ctx.setLineDash([]);

                coordinates.forEach((coord, index) => {
                    if (index < coordinates.length - 1) {
                        const point1 = mapToCanvas(coord.lat, coord.lng);
                        const point2 = mapToCanvas(coordinates[index + 1].lat, coordinates[index + 1].lng);

                        ctx.beginPath();
                        ctx.moveTo(point1.x, point1.y);
                        ctx.lineTo(point2.x, point2.y);
                        ctx.stroke();
                    }
                });

                // Draw closing line
                if (coordinates.length > 2) {
                    const firstPoint = mapToCanvas(coordinates[0].lat, coordinates[0].lng);
                    const lastPoint = mapToCanvas(coordinates[coordinates.length - 1].lat, coordinates[coordinates.length - 1].lng);

                    ctx.beginPath();
                    ctx.moveTo(lastPoint.x, lastPoint.y);
                    ctx.lineTo(firstPoint.x, firstPoint.y);
                    ctx.stroke();
                }

                // Draw markers with shadows
                coordinates.forEach((coord, index) => {
                    const point = mapToCanvas(coord.lat, coord.lng);

                    // Draw shadow
                    ctx.fillStyle = 'rgba(0, 0, 0, 0.2)';
                    ctx.beginPath();
                    ctx.arc(point.x + 2, point.y + 2, 16, 0, 2 * Math.PI);
                    ctx.fill();

                    // Draw marker circle
                    ctx.fillStyle = '#28a745';
                    ctx.strokeStyle = '#ffffff';
                    ctx.lineWidth = 3;
                    ctx.setLineDash([]);

                    ctx.beginPath();
                    ctx.arc(point.x, point.y, 16, 0, 2 * Math.PI);
                    ctx.fill();
                    ctx.stroke();

                    // Draw number
                    ctx.fillStyle = '#ffffff';
                    ctx.font = 'bold 14px Arial';
                    ctx.textAlign = 'center';
                    ctx.textBaseline = 'middle';
                    ctx.fillText((index + 1).toString(), point.x, point.y);

                    // Add coordinate label
                    ctx.fillStyle = '#495057';
                    ctx.font = '9px Arial';
                    ctx.textAlign = 'center';
                    ctx.fillText(
                        `${coord.lat.toFixed(6)}, ${coord.lng.toFixed(6)}`,
                        point.x,
                        point.y + 30
                    );
                });

                // Add border
                ctx.strokeStyle = '#495057';
                ctx.lineWidth = 2;
                ctx.setLineDash([]);
                ctx.strokeRect(2, 2, canvas.width - 4, canvas.height - 4);
            }

            // Add header with title
            ctx.fillStyle = '#ffffff';
            ctx.fillRect(0, 0, canvas.width, 45);
            ctx.strokeStyle = '#dee2e6';
            ctx.lineWidth = 1;
            ctx.strokeRect(0, 0, canvas.width, 45);

            ctx.fillStyle = '#212529';
            ctx.font = 'bold 18px Arial';
            ctx.textAlign = 'center';
            ctx.fillText('PETA KOORDINAT PENGUKURAN BIDANG TANAH', canvas.width / 2, 28);

            // Add footer with information
            ctx.fillStyle = '#ffffff';
            ctx.fillRect(0, canvas.height - 50, canvas.width, 50);
            ctx.strokeRect(0, canvas.height - 50, canvas.width, 50);

            // Left side info
            ctx.fillStyle = '#495057';
            ctx.font = 'bold 12px Arial';
            ctx.textAlign = 'left';
            ctx.fillText(`Luas Area: ${calculatedArea.toLocaleString('id-ID')} m²`, 20, canvas.height - 30);
            ctx.fillText(`Jumlah Titik: ${coordinates.length}`, 20, canvas.height - 15);

            // Center info
            ctx.textAlign = 'center';
            ctx.fillText('Sistem Koordinat: UTM', canvas.width / 2, canvas.height - 30);
            ctx.fillText(`Skala: 1:${Math.round(1000 + coordinates.length * 100)}`, canvas.width / 2, canvas.height - 15);

            // Right side info
            ctx.textAlign = 'right';
            ctx.fillText(`Tanggal: ${new Date().toLocaleDateString('id-ID')}`, canvas.width - 20, canvas.height - 30);
            ctx.fillText(`Waktu: ${new Date().toLocaleTimeString('id-ID')}`, canvas.width - 20, canvas.height - 15);

            // Save the image
            const imageData = canvas.toDataURL('image/png');
            document.getElementById('map_image_data').value = imageData;

            if (saveStatus) {
                saveStatus.textContent = 'Peta berhasil di-capture!';
                saveStatus.className = 'text-success';
            }

            // Optional: Show preview in console for debugging
            console.log('Map captured successfully, image size:', imageData.length, 'bytes');

        } catch (error) {
            console.error('Error creating manual map image:', error);

            if (saveStatus) {
                saveStatus.textContent = 'Error capture peta';
                saveStatus.className = 'text-danger';
            }

            // Simplified fallback
            fallbackCapture();
        }
    }

    // Alternative capture method with html2canvas (kept as backup)
    function captureMapAlternative() {
        const saveStatus = document.getElementById('saveStatus');

        try {
            const mapContainer = document.getElementById('pengukuranMap');

            if (typeof html2canvas !== 'undefined') {
                // Force map to redraw first
                if (pengukuranMap) {
                    pengukuranMap.invalidateSize();

                    setTimeout(function() {
                        html2canvas(mapContainer, {
                            useCORS: true,
                            allowTaint: true,
                            scale: 1,
                            width: 800,
                            height: 600,
                            backgroundColor: '#ffffff',
                            logging: false
                        }).then(function(canvas) {
                            const imageData = canvas.toDataURL('image/png');
                            document.getElementById('map_image_data').value = imageData;

                            if (saveStatus) {
                                saveStatus.textContent = 'Peta berhasil di-capture (html2canvas)';
                                saveStatus.className = 'text-success';
                            }
                        }).catch(function(error) {
                            console.error('html2canvas failed:', error);
                            createManualMapImage();
                        });
                    }, 500);
                }
            } else {
                createManualMapImage();
            }
        } catch (error) {
            console.error('Error in alternative capture:', error);
            createManualMapImage();
        }
    }

    // Fallback capture method
    function fallbackCapture() {
        try {
            const mapData = {
                coordinates: coordinates,
                bounds: pengukuranPolygon ? pengukuranPolygon.getBounds() : null,
                center: pengukuranMap ? pengukuranMap.getCenter() : null,
                zoom: pengukuranMap ? pengukuranMap.getZoom() : 18,
                calculatedArea: calculatedArea,
                timestamp: new Date().toISOString()
            };

            document.getElementById('map_image_data').value = JSON.stringify(mapData);

            const saveStatus = document.getElementById('saveStatus');
            if (saveStatus) {
                saveStatus.textContent = 'Data peta berhasil disimpan';
                saveStatus.className = 'text-success';
            }
        } catch (error) {
            console.error('Error in fallback capture:', error);
            const saveStatus = document.getElementById('saveStatus');
            if (saveStatus) {
                saveStatus.textContent = 'Error menyimpan data peta';
                saveStatus.className = 'text-danger';
            }
        }
    }

    // Auto-capture when coordinates change (debounced)
    let captureTimeout;
    function autoCapture() {
        if (coordinates.length >= 3 && calculatedArea > 0) {
            if (captureTimeout) {
                clearTimeout(captureTimeout);
            }

            captureTimeout = setTimeout(function() {
                captureMap();
            }, 2000);
        }
    }

    // Initialize when document ready and tab is shown
    $(document).ready(function() {
        // Initialize map when pengukuran tab is shown
        $('a[href="#pengukuran"]').on('shown.bs.tab', function() {
            setTimeout(function() {
                if (!mapInitialized) {
                    initPengukuranMap();
                } else {
                    updateMapAndCalculate();
                }
            }, 100);
        });

        // Handle online/offline status
        window.addEventListener('online', function() {
            console.log('Connection restored');
            if (pengukuranMap && mapInitialized) {
                pengukuranMap.eachLayer(function(layer) {
                    if (layer.options && layer.options.attribution) {
                        layer.redraw();
                    }
                });
            }
        });

        window.addEventListener('offline', function() {
            console.log('Connection lost');
        });

        // Auto-save coordinates when they change
        $(document).on('change', 'input[name^="lat"], input[name^="lng"]', function() {
            setTimeout(function() {
                updateMapAndCalculate();
                autoCapture();
            }, 500);
        });
    });

    // Ensure all required elements exist
    function ensureElements() {
        // Ensure hidden inputs exist
        const requiredInputs = [
            'coordinates_input',
            'luas_terukur_otomatis',
            'map_image_data',
            'polygon_bounds'
        ];

        requiredInputs.forEach(inputId => {
            if (!document.getElementById(inputId)) {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.id = inputId;
                input.name = inputId.replace('_input', '_data');

                const form = document.querySelector('form');
                if (form) {
                    form.appendChild(input);
                }
            }
        });

        // Ensure status elements exist
        const statusElements = [
            'calculatedArea',
            'coordinateCount',
            'areaStatus',
            'areaProgress',
            'saveStatus'
        ];

        statusElements.forEach(elementId => {
            if (!document.getElementById(elementId)) {
                console.warn(`Missing status element: ${elementId}`);
            }
        });
    }

    // Call ensure elements when document ready
    $(document).ready(function() {
        ensureElements();

        // Initialize with existing coordinates if any
        setTimeout(function() {
            const existingRows = document.querySelectorAll('#coordinatesList tr');
            if (existingRows.length >= 3) {
                updateMapAndCalculate();
            }
        }, 1000);
    });

    // Export functions for external access
    window.initPengukuranMap = initPengukuranMap;
    window.updateMapAndCalculate = updateMapAndCalculate;
    window.addCoordinate = addCoordinate;
    window.removeCoordinate = removeCoordinate;
    window.centerMap = centerMap;
    window.captureMap = captureMap;
    window.validateCoordinateInput = validateCoordinateInput;
</script>
@endpush
