@extends('admin-layouts.master')

@section('content')
<div class="container-fluid">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Tambah Layanan Baru</h1>
        <a href="{{ route('services.index') }}" class="btn btn-secondary">
            <i class="fas fa-arrow-left"></i> Kembali
        </a>
    </div>

    <!-- Form Card -->
    <div class="row">
        <div class="col-lg-8">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Form Layanan</h6>
                </div>
                <div class="card-body">
                    <form action="{{ route('services.store') }}" method="POST" id="serviceForm">
                        @csrf
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="name">Nama Layanan <span class="text-danger">*</span></label>
                                    <input type="text" 
                                           class="form-control @error('name') is-invalid @enderror" 
                                           id="name" 
                                           name="name" 
                                           value="{{ old('name') }}" 
                                           placeholder="Contoh: Balinese Massage, Room Service, Laundry"
                                           required>
                                    @error('name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="service_category_id">Kategori Layanan <span class="text-danger">*</span></label>
                                    <select class="form-control @error('service_category_id') is-invalid @enderror" 
                                            id="service_category_id" 
                                            name="service_category_id" 
                                            required>
                                        <option value="">Pilih Kategori</option>
                                        @foreach($categories as $category)
                                            <option value="{{ $category->id }}" {{ old('service_category_id') == $category->id ? 'selected' : '' }}>
                                                {{ $category->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('service_category_id')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="satuan">Satuan <span class="text-danger">*</span></label>
                                    <input type="text" 
                                           class="form-control @error('satuan') is-invalid @enderror" 
                                           id="satuan" 
                                           name="satuan" 
                                           value="{{ old('satuan') }}" 
                                           placeholder="Contoh: per jam, per porsi, per kg"
                                           required>
                                    @error('satuan')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    <small class="form-text text-muted">
                                        Unit pengukuran untuk layanan ini
                                    </small>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="harga">Harga (Rp) <span class="text-danger">*</span></label>
                                    <input type="number" 
                                           class="form-control @error('harga') is-invalid @enderror" 
                                           id="harga" 
                                           name="harga" 
                                           value="{{ old('harga') }}" 
                                           placeholder="0" 
                                           min="0" 
                                           step="1000"
                                           required>
                                    @error('harga')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    <small class="form-text text-muted">
                                        Harga per satuan sebelum PPN
                                    </small>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="ppn">PPN (%) <span class="text-danger">*</span></label>
                                    <input type="number" 
                                           class="form-control @error('ppn') is-invalid @enderror" 
                                           id="ppn" 
                                           name="ppn" 
                                           value="{{ old('ppn', 10) }}" 
                                           placeholder="10" 
                                           min="0" 
                                           max="100" 
                                           step="0.1"
                                           required>
                                    @error('ppn')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    <small class="form-text text-muted">
                                        Persentase PPN (biasanya 10%)
                                    </small>
                                </div>
                            </div>
                        </div>

                        <!-- Price Preview -->
                        <div class="card bg-light mb-3" id="pricePreview" style="display: none;">
                            <div class="card-body">
                                <h6 class="card-title">Preview Harga</h6>
                                <div class="row">
                                    <div class="col-md-6">
                                        <table class="table table-borderless table-sm">
                                            <tr>
                                                <td>Harga Dasar:</td>
                                                <td class="text-right" id="previewBasePrice">Rp 0</td>
                                            </tr>
                                            <tr>
                                                <td>PPN (<span id="previewPpnRate">0</span>%):</td>
                                                <td class="text-right" id="previewPpnAmount">Rp 0</td>
                                            </tr>
                                        </table>
                                    </div>
                                    <div class="col-md-6">
                                        <table class="table table-borderless table-sm">
                                            <tr class="font-weight-bold text-success">
                                                <td>Harga Final:</td>
                                                <td class="text-right" id="previewFinalPrice">Rp 0</td>
                                            </tr>
                                            <tr>
                                                <td>Per:</td>
                                                <td class="text-right" id="previewUnit">-</td>
                                            </tr>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <button type="submit" class="btn btn-success">
                                <i class="fas fa-save"></i> Simpan Layanan
                            </button>
                            <a href="{{ route('services.index') }}" class="btn btn-secondary">
                                <i class="fas fa-times"></i> Batal
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-4">
            <!-- Help Card -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-info">Panduan Membuat Layanan</h6>
                </div>
                <div class="card-body">
                    <h6 class="text-primary">Tips:</h6>
                    <ul class="text-sm">
                        <li>Gunakan nama layanan yang jelas dan spesifik</li>
                        <li>Pilih kategori yang sesuai dengan jenis layanan</li>
                        <li>Tentukan satuan yang mudah dipahami</li>
                        <li>Set harga competitif dengan market</li>
                        <li>PPN biasanya 10% untuk layanan hotel</li>
                    </ul>
                    
                    <hr>
                    
                    <h6 class="text-primary">Contoh Layanan:</h6>
                    <div class="mb-2">
                        <strong>Spa & Wellness:</strong>
                        <small class="d-block text-muted">Balinese Massage - Rp 300.000/60 menit</small>
                    </div>
                    <div class="mb-2">
                        <strong>Food & Beverage:</strong>
                        <small class="d-block text-muted">Nasi Gudeg Set - Rp 85.000/porsi</small>
                    </div>
                    <div class="mb-2">
                        <strong>Laundry:</strong>
                        <small class="d-block text-muted">Cuci Setrika - Rp 8.000/kg</small>
                    </div>
                    <div class="mb-2">
                        <strong>Transportation:</strong>
                        <small class="d-block text-muted">Airport Shuttle - Rp 150.000/trip</small>
                    </div>
                </div>
            </div>

            <!-- Quick Stats -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-success">Statistik Layanan</h6>
                </div>
                <div class="card-body">
                    <div class="text-center mb-3">
                        <div class="h4 mb-0 font-weight-bold text-primary">
                            {{ \App\Models\MsService::count() }}
                        </div>
                        <div class="text-xs text-gray-600">Total Layanan</div>
                    </div>
                    
                    <hr>
                    
                    @foreach($categories as $category)
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <span class="text-sm">{{ $category->name }}</span>
                        <span class="badge badge-primary">{{ $category->services->count() }}</span>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
$(document).ready(function() {
    // Auto focus on name input
    $('#name').focus();
    
    // Calculate price preview when inputs change
    $('#harga, #ppn, #satuan').on('input', function() {
        updatePricePreview();
    });
    
    function updatePricePreview() {
        const basePrice = parseFloat($('#harga').val()) || 0;
        const ppnRate = parseFloat($('#ppn').val()) || 0;
        const unit = $('#satuan').val() || '';
        
        if (basePrice > 0) {
            const ppnAmount = (basePrice * ppnRate) / 100;
            const finalPrice = basePrice + ppnAmount;
            
            $('#previewBasePrice').text(formatRupiah(basePrice));
            $('#previewPpnRate').text(ppnRate);
            $('#previewPpnAmount').text(formatRupiah(ppnAmount));
            $('#previewFinalPrice').text(formatRupiah(finalPrice));
            $('#previewUnit').text(unit || 'satuan');
            
            $('#pricePreview').show();
        } else {
            $('#pricePreview').hide();
        }
    }
    
    function formatRupiah(amount) {
        return 'Rp ' + new Intl.NumberFormat('id-ID').format(amount);
    }
    
    // Form validation
    $('#serviceForm').on('submit', function(e) {
        const name = $('#name').val().trim();
        const harga = parseFloat($('#harga').val()) || 0;
        const ppn = parseFloat($('#ppn').val()) || 0;
        
        if (name.length < 3) {
            e.preventDefault();
            Swal.fire({
                icon: 'warning',
                title: 'Nama Terlalu Pendek',
                text: 'Nama layanan minimal 3 karakter'
            });
            return false;
        }
        
        if (harga <= 0) {
            e.preventDefault();
            Swal.fire({
                icon: 'warning',
                title: 'Harga Tidak Valid',
                text: 'Harga harus lebih dari 0'
            });
            return false;
        }
        
        if (ppn < 0 || ppn > 100) {
            e.preventDefault();
            Swal.fire({
                icon: 'warning',
                title: 'PPN Tidak Valid',
                text: 'PPN harus antara 0-100%'
            });
            return false;
        }
    });
    
    // Auto-suggest common units based on category
    $('#service_category_id').on('change', function() {
        const categoryText = $(this).find('option:selected').text().toLowerCase();
        const satuanInput = $('#satuan');
        
        let suggestedUnit = '';
        
        if (categoryText.includes('spa') || categoryText.includes('wellness')) {
            suggestedUnit = 'per sesi';
        } else if (categoryText.includes('food') || categoryText.includes('beverage')) {
            suggestedUnit = 'per porsi';
        } else if (categoryText.includes('laundry')) {
            suggestedUnit = 'per kg';
        } else if (categoryText.includes('transport')) {
            suggestedUnit = 'per trip';
        }
        
        if (suggestedUnit && !satuanInput.val()) {
            satuanInput.val(suggestedUnit);
            updatePricePreview();
        }
    });
});
</script>
@endpush
@endsection