@extends('admin-layouts.master')

@section('title', 'SP-MPT Baru')

@push('styles')
<style>
    body {
        background-color: #f8f9fa;
    }

    .header-gradient {
        background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
        color: white;
        padding: 1rem;
        border-radius: 0;
    }

    .card {
        border: none;
        box-shadow: 0 0.125rem 0.25rem rgba(0,0,0,0.075);
        margin-bottom: 1.5rem;
    }

    .section-title {
        font-size: 0.875rem;
        font-weight: 600;
        text-transform: uppercase;
        color: #495057;
        margin-bottom: 1rem;
        padding-bottom: 0.5rem;
        border-bottom: 2px solid #dee2e6;
    }

    .form-control:focus {
        border-color: #28a745;
        box-shadow: 0 0 0 0.2rem rgba(40, 167, 69, 0.25);
    }

    .required {
        color: #dc3545;
    }

    .upload-area {
        border: 2px dashed #dee2e6;
        border-radius: 0.25rem;
        padding: 2rem;
        text-align: center;
        cursor: pointer;
        transition: all 0.3s;
        background-color: #fafafa;
    }

    .upload-area:hover {
        background-color: #e9ecef;
        border-color: #28a745;
    }

    .floating-buttons {
        position: fixed;
        bottom: 1rem;
        right: 1rem;
        z-index: 1000;
    }

    .btn-circle {
        width: 3rem;
        height: 3rem;
        border-radius: 50%;
        margin: 0.25rem;
        box-shadow: 0 0.25rem 0.5rem rgba(0,0,0,0.15);
    }

    #polygonMap {
        height: 450px;
        width: 100%;
        border: 1px solid #dee2e6;
        border-radius: 0.25rem;
    }

    .coordinate-input {
        font-size: 0.875rem;
    }

    @media print {
        .no-print {
            display: none !important;
        }

        .card {
            box-shadow: none;
            border: 1px solid #dee2e6;
        }
    }
</style>

<!-- Leaflet CSS -->
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" integrity="sha256-p4NxAoJBhIIN+hmNHrzRCf9tD/miZyoHS5obTRR9BMY=" crossorigin="" />
<!-- Proj4js for UTM conversion -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/proj4js/2.7.4/proj4.js"></script>
@endpush

@section('content')
<div class="container-fluid">
    <!-- Header -->
    <div class="card">
        <div class="header-gradient">
            <h5 class="mb-0 font-weight-bold">SP-MPT BARU</h5>
            <small>Surat Pernyataan Melepaskan Penguasaan Tanah</small>
        </div>
    </div>

    <!-- Alerts -->
    @if(session('success'))
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <i class="fas fa-check-circle mr-2"></i>{{ session('success') }}
        <button type="button" class="close" data-dismiss="alert">
            <span>&times;</span>
        </button>
    </div>
    @endif

    @if($errors->any())
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <strong><i class="fas fa-exclamation-triangle mr-2"></i>Terdapat kesalahan:</strong>
        <ul class="mb-0 mt-2">
            @foreach($errors->all() as $error)
                <li>{{ $error }}</li>
            @endforeach
        </ul>
        <button type="button" class="close" data-dismiss="alert">
            <span>&times;</span>
        </button>
    </div>
    @endif

    <!-- Main Form -->
    <form method="POST" action="{{ route('sp-sipt.store') }}" enctype="multipart/form-data" id="formSpSipt">
        @csrf

        <div class="card">
            <div class="card-body">
                <!-- Nomor Surat -->
                <div class="form-group">
                    <label class="font-weight-bold">Nomor Surat:</label>
                    <input type="text" class="form-control bg-light" value="Otomatis saat disimpan" readonly>
                    <small class="text-muted">*Nomor surat akan digenerate otomatis</small>
                </div>
                <!-- PIHAK PERTAMA -->
                <h6 class="section-title">PIHAK PERTAMA</h6>

                <div class="form-row">
                    <div class="form-group col-md-6">
                        <label>Nama <span class="required">*</span></label>
                        <input type="text"
                               class="form-control @error('pemohon_nama') is-invalid @enderror"
                               name="pemohon_nama"
                               value="{{ old('pemohon_nama') }}"
                               placeholder="Masukkan nama lengkap"
                               required>
                        @error('pemohon_nama')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="form-group col-md-6">
                        <label>Tempat/Tanggal Lahir</label>
                        <div class="input-group">
                            <input type="text"
                                   class="form-control @error('pemohon_tempat_lahir') is-invalid @enderror"
                                   name="pemohon_tempat_lahir"
                                   value="{{ old('pemohon_tempat_lahir') }}"
                                   placeholder="Tempat lahir">
                            <input type="date"
                                   class="form-control @error('pemohon_tanggal_lahir') is-invalid @enderror"
                                   name="pemohon_tanggal_lahir"
                                   value="{{ old('pemohon_tanggal_lahir') }}">
                        </div>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group col-md-6">
                        <label>Nomor KTP <span class="required">*</span></label>
                        <input type="text"
                               class="form-control @error('pemohon_nik') is-invalid @enderror"
                               name="pemohon_nik"
                               value="{{ old('pemohon_nik') }}"
                               placeholder="16 digit NIK"
                               maxlength="16"
                               pattern="[0-9]{16}"
                               required>
                        @error('pemohon_nik')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="form-group col-md-6">
                        <label>Pekerjaan</label>
                        <input type="text"
                               class="form-control @error('pemohon_pekerjaan') is-invalid @enderror"
                               name="pemohon_pekerjaan"
                               value="{{ old('pemohon_pekerjaan') }}"
                               placeholder="Masukkan pekerjaan">
                    </div>
                </div>

                <div class="form-group">
                    <label>Alamat <span class="required">*</span></label>
                    <textarea class="form-control @error('pemohon_alamat') is-invalid @enderror"
                              name="pemohon_alamat"
                              rows="2"
                              placeholder="Masukkan alamat lengkap"
                              required>{{ old('pemohon_alamat') }}</textarea>
                    @error('pemohon_alamat')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>

                <!-- PIHAK KEDUA -->
                <h6 class="section-title mt-4">PIHAK KEDUA</h6>

                <div class="form-row">
                    <div class="form-group col-md-6">
                        <label>Nama <span class="required">*</span></label>
                        <input type="text"
                               class="form-control @error('terlapor_nama') is-invalid @enderror"
                               name="terlapor_nama"
                               value="{{ old('terlapor_nama') }}"
                               placeholder="Masukkan nama lengkap"
                               required>
                        @error('terlapor_nama')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="form-group col-md-6">
                        <label>Tempat/Tanggal Lahir</label>
                        <div class="input-group">
                            <input type="text"
                                   class="form-control @error('terlapor_tempat_lahir') is-invalid @enderror"
                                   name="terlapor_tempat_lahir"
                                   value="{{ old('terlapor_tempat_lahir') }}"
                                   placeholder="Tempat lahir">
                            <input type="date"
                                   class="form-control @error('terlapor_tanggal_lahir') is-invalid @enderror"
                                   name="terlapor_tanggal_lahir"
                                   value="{{ old('terlapor_tanggal_lahir') }}">
                        </div>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group col-md-6">
                        <label>Nomor KTP</label>
                        <input type="text"
                               class="form-control @error('terlapor_nik') is-invalid @enderror"
                               name="terlapor_nik"
                               value="{{ old('terlapor_nik') }}"
                               placeholder="16 digit NIK"
                               maxlength="16"
                               pattern="[0-9]{16}">
                        @error('terlapor_nik')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="form-group col-md-6">
                        <label>Pekerjaan</label>
                        <input type="text"
                               class="form-control @error('terlapor_pekerjaan') is-invalid @enderror"
                               name="terlapor_pekerjaan"
                               value="{{ old('terlapor_pekerjaan') }}"
                               placeholder="Masukkan pekerjaan">
                    </div>
                </div>

                <div class="form-group">
                    <label>Alamat</label>
                    <textarea class="form-control @error('terlapor_alamat') is-invalid @enderror"
                              name="terlapor_alamat"
                              rows="2"
                              placeholder="Masukkan alamat lengkap">{{ old('terlapor_alamat') }}</textarea>
                    @error('terlapor_alamat')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>
                <!-- DATA TANAH -->
                <h6 class="section-title mt-4">DATA TANAH</h6>
                <div class="form-row">
                    <div class="form-group col-md-4">
                        <label>Ketua RT</label>
                        <input type="text"
                               class="form-control @error('ketua_rt') is-invalid @enderror"
                               name="ketua_rt"
                               value="{{ old('ketua_rt') }}"
                               placeholder="">
                        @error('ketua_rt')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    <div class="form-group col-md-4">
                        <label>Nama RT</label>
                        <input type="text"
                               class="form-control @error('nama_rt') is-invalid @enderror"
                               name="nama_rt"
                               value="{{ old('nama_rt') }}"
                               placeholder="">
                        @error('nama_rt')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    <div class="form-group col-md-4">
                        <label>Lokasi Tanah RT</label>
                        <input type="text"
                               class="form-control @error('lokasi_tanah_rt') is-invalid @enderror"
                               name="lokasi_tanah_rt"
                               value="{{ old('lokasi_tanah_rt') }}"
                               placeholder="">
                        @error('lokasi_tanah_rt')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group col-md-4">
                        <label for="Kampung">Kampung/Kelurahan</label>
                        <select class="form-control select2" id="kampung" name="kampung" required>
                            <option value="">-- Pilih Kampung --</option>
                            @foreach($kampungs as $kampung)
                                <option value="{{ $kampung->id }}"
                                    {{ old('kampung') == $kampung->id ? 'selected' : '' }}>
                                    {{ $kampung->nama_kampung }}
                                </option>
                            @endforeach
                        </select>
                        @error('kampung')
                            <small class="text-danger">{{ $message }}</small>
                        @enderror
                    </div>
                    <div class="form-group col-md-4">
                        <label for="Kecamatan">Kecamatan</label>
                        <select class="form-control select2" id="kecamatan_id" name="kecamatan_id" required>
                            <option value="">-- Pilih Kecamatan --</option>
                            @foreach($kecamatans as $kecamatan)
                                <option value="{{ $kecamatan->id }}"
                                    {{ old('kecamatan_id') == $kecamatan->id ? 'selected' : '' }}>
                                    {{ $kecamatan->nama_kecamatan }}
                                </option>
                            @endforeach
                        </select>
                        @error('kecamatan_id')
                            <small class="text-danger">{{ $message }}</small>
                        @enderror
                    </div>
                    <div class="form-group col-md-4">
                        <label>Kota/Kabupaten</label>
                        <input type="text"
                               class="form-control @error('lokasi_kota') is-invalid @enderror"
                               name="lokasi_kota"
                               value="{{ old('lokasi_kota') }}"
                               placeholder="Nama kota">
                        @error('lokasi_kota')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>

                <!-- BATAS-BATAS TANAH -->
                <div class="form-row">
                    <div class="form-group col-md-6">
                        <label>Utara</label>
                        <input type="text"
                               class="form-control @error('batas_utara') is-invalid @enderror"
                               name="batas_utara"
                               value="{{ old('batas_utara') }}"
                               placeholder="Berbatasan dengan...">
                        @error('batas_utara')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="form-group col-md-6">
                        <label>Timur</label>
                        <input type="text"
                               class="form-control @error('batas_timur') is-invalid @enderror"
                               name="batas_timur"
                               value="{{ old('batas_timur') }}"
                               placeholder="Berbatasan dengan...">
                        @error('batas_timur')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group col-md-6">
                        <label>Selatan</label>
                        <input type="text"
                               class="form-control @error('batas_selatan') is-invalid @enderror"
                               name="batas_selatan"
                               value="{{ old('batas_selatan') }}"
                               placeholder="Berbatasan dengan...">
                        @error('batas_selatan')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="form-group col-md-6">
                        <label>Barat</label>
                        <input type="text"
                               class="form-control @error('batas_barat') is-invalid @enderror"
                               name="batas_barat"
                               value="{{ old('batas_barat') }}"
                               placeholder="Berbatasan dengan...">
                        @error('batas_barat')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>

                <!-- PERNYATAAN -->
                <h6 class="section-title mt-4"></h6>

                <div class="form-group">
                    <div class="form-group col-md-4">
                        <label>Bukti (Kwitansi) Pembelian Tanggal</label>
                        <input type="date"
                               class="form-control @error('tanggal_sertifikat') is-invalid @enderror"
                               name="tanggal_sertifikat"
                               value="{{ old('tanggal_sertifikat') }}"
                               placeholder="Berbatasan dengan...">
                        @error('tanggal_sertifikat')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>

                <div class="form-group">
                    <div class="form-group col-md-4">
                        <label>Nomor dan Tanggal Surat Keterangan Penguasaan dan Pemilikan Bangunan/Tanaman Diatas Tanah Negara:</label>
                        <textarea name="nomor_tanggal_surat_penguasaan_tanah" class="form-control @error('nomor_tanggal_surat_penguasaan_tanah') is-invalid @enderror" id="" cols="30" rows="3">
                            {{ old('nomor_tanggal_surat_penguasaan_tanah') }}
                        </textarea>
                        @error('nomor_tanggal_surat_penguasaan_tanah')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
                <!-- UPLOAD DOKUMEN -->
                <h6 class="section-title mt-4">UPLOAD DOKUMEN</h6>

                <div class="form-row">
                    <div class="form-group col-md-6">
                        <label>Upload KWITANSI (PDF/IMAGE)</label>
                        <div class="upload-area" onclick="document.getElementById('ktp_file').click()">
                            <i class="fas fa-cloud-upload-alt fa-3x text-muted mb-3"></i>
                            <p class="mb-1">Klik untuk upload file berformat PDF</p>
                            <small class="text-muted">Format: PDF • Maksimal: 5MB</small>
                            <input type="file"
                                   id="ktp_file"
                                   name="kwitansi"
                                   class="d-none"
                                   accept=".pdf,image/*"
                                   onchange="showFileInfo(this, 'ktp_preview')">
                            <div id="ktp_preview" class="mt-3"></div>
                        </div>
                        @error('kwitansi')
                            <small class="text-danger">{{ $message }}</small>
                        @enderror
                    </div>

                    <div class="form-group col-md-6">
                        <label>Upload Dokumen Pendukung</label>
                        <div class="upload-area" onclick="document.getElementById('template_file').click()">
                            <i class="fas fa-cloud-upload-alt fa-3x text-muted mb-3"></i>
                            <p class="mb-1">Klik untuk upload dokumen template</p>
                            <small class="text-muted">Format: PDF, DOC, DOCX • Maksimal: 10MB</small>
                            <input type="file"
                                   id="template_file"
                                   name="dokumen_template"
                                   class="d-none"
                                   accept=".pdf,.doc,.docx"
                                   onchange="showFileInfo(this, 'template_preview')">
                            <div id="template_preview" class="mt-3"></div>
                        </div>
                        @error('dokumen_template')
                            <small class="text-danger">{{ $message }}</small>
                        @enderror
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group col-md-6">
                        <label>Peruntukan Bidang Tanah</label>
                        <input type="text"
                               class="form-control @error('peruntukan_bidang_tanah') is-invalid @enderror"
                               name="peruntukan_bidang_tanah"
                               value="{{ old('peruntukan_bidang_tanah') }}"
                               placeholder="">
                        @error('peruntukan_bidang_tanah')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="form-group col-md-6">
                        <label>Uang Ganti Kerugian/Santunan</label>
                        <input type="text"
                               class="form-control @error('uang_ganti_kerugian') is-invalid @enderror"
                               name="uang_ganti_kerugian"
                               value="{{ old('uang_ganti_kerugian') }}"
                               placeholder="">
                        @error('uang_ganti_kerugian')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
                <!-- KOORDINAT POLYGON Section -->
                <h6 class="section-title mt-4">KOORDINAT POLYGON TANAH</h6>

                <div class="row">
                    <div class="col-lg-5">
                        <!-- Coordinate Input Table -->
                        <div class="card border-primary mb-3">
                            <div class="card-header bg-primary text-white">
                                <h6 class="mb-0">Input Koordinat UTM (Zone 50N)</h6>
                            </div>
                            <div class="card-body" style="max-height: 400px; overflow-y: auto;">
                                <table class="table table-sm">
                                    <thead class="bg-light sticky-top">
                                        <tr>
                                            <th width="10%">#</th>
                                            <th width="35%">Easting (m)</th>
                                            <th width="35%">Northing (m)</th>
                                            <th width="20%" class="text-center">Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody id="coordinatesList">
                                        <!-- Rows will be dynamically generated by loadExistingCoordinates() - 100% match edit-tab-pengukuran.blade.php -->
                                    </tbody>
                                </table>

                                <button type="button" class="btn btn-primary btn-sm btn-block" onclick="addCoordinate()">
                                    <i class="fas fa-plus"></i> Tambah Koordinat
                                </button>

                                <!-- Hidden inputs for coordinate data -->
                                <input type="hidden" name="coordinates_data" id="coordinates_data" value="{{ old('coordinates_data') }}">
                                <input type="hidden" name="map_image_data" id="map_image_data" value="{{ old('map_image_data') }}">
                                <input type="hidden" name="luas_terukur" id="luas_terukur" value="{{ old('luas_terukur') }}">
                                <!-- Hidden inputs for lat/lng conversions -->
                                <div id="latLngHiddenInputs"></div>
                            </div>
                        </div>

                        <!-- Area Information -->
                        <div class="card border-success">
                            <div class="card-header bg-success text-white">
                                <h6 class="mb-0">Informasi Area</h6>
                            </div>
                            <div class="card-body">
                                <div class="row text-center">
                                    <div class="col-6">
                                        <div class="h4 text-success mb-0" id="calculatedArea">0 m²</div>
                                        <small class="text-muted">Luas Terukur</small>
                                    </div>
                                    <div class="col-6">
                                        <div class="h4 text-info mb-0" id="coordinateCount">0</div>
                                        <small class="text-muted">Titik Koordinat</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-lg-7">
                        <!-- Map with Polygon -->
                        <div class="card border-info">
                            <div class="card-header bg-info text-white d-flex justify-content-between">
                                <h6 class="mb-0">Peta Polygon</h6>
                                <div class="btn-group btn-group-sm">
                                    <button type="button" class="btn btn-light btn-sm" onclick="centerMap()">
                                        <i class="fas fa-crosshairs"></i> Center
                                    </button>
                                    <button type="button" class="btn btn-warning btn-sm" onclick="captureMap(true)">
                                        <i class="fas fa-camera"></i> Capture
                                    </button>
                                </div>
                            </div>
                            <div class="card-body p-0">
                                <div id="polygonMap" style="height: 450px;"></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Floating Buttons -->
        <div class="floating-buttons no-print">
            <button type="submit" class="btn btn-success btn-circle" title="Simpan">
                <i class="fas fa-save"></i>
            </button>
            <button type="button" class="btn btn-secondary btn-circle" onclick="window.location.href='{{ route('sp-sipt.index') }}'" title="Kembali">
                <i class="fas fa-arrow-left"></i>
            </button>
        </div>
    </form>
</div>
@endsection

@push('scripts')
<!-- Leaflet JS -->
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js" integrity="sha256-20nQCchB9co0qIjJZRGuk2/Z9VM+kNiyxNV1lvTlZBo=" crossorigin=""></script>
<!-- jQuery and Bootstrap JS -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<!-- Select2 JS -->
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />

<script>
// ============================================
// UTM COORDINATE CONVERSION SETUP
// ============================================

// Define UTM Zone 50N (for Kalimantan Timur - Berau area)
// EPSG:32650 = WGS 84 / UTM zone 50N
proj4.defs("EPSG:32650", "+proj=utm +zone=50 +datum=WGS84 +units=m +no_defs");

// Helper function to convert UTM to Lat/Lng (EXACT copy from edit-tab-pengukuran)
function utmToLatLng(easting, northing) {
    // Relaxed validation: accept any positive easting/northing
    easting = parseFloat(easting);
    northing = parseFloat(northing);
    if (isNaN(easting) || isNaN(northing) || easting <= 0 || northing <= 0) {
        console.error(`Invalid UTM coordinate: Easting ${easting}, Northing ${northing}`);
        return null;
    }
    try {
        // Use EPSG:32650 definition for Zone 50N
        const [lng, lat] = proj4('EPSG:32650', '+proj=longlat +datum=WGS84 +no_defs', [easting, northing]);
        console.log(`UTM (${easting}, ${northing}) → Lat/Lng (${lat}, ${lng})`);
        return { lat: lat, lng: lng };
    } catch (error) {
        console.error('Error converting UTM to Lat/Lng:', error);
        return null;
    }
}

// Helper function to convert Lat/Lng to UTM (Zone 50N ONLY)
function latLngToUTM(lat, lng) {
    try {
        const [easting, northing] = proj4('+proj=longlat +datum=WGS84 +no_defs', 'EPSG:32650', [lng, lat]);
        return { easting: easting, northing: northing };
    } catch (error) {
        console.error('Error converting Lat/Lng to UTM:', error);
        return null;
    }
}

// ============================================
// GLOBAL VARIABLES (100% match edit-tab-pengukuran)
// ============================================

// Global variables for map and coordinates
let map = null;
let polygon = null;
let markers = [];
let coordinateCount = 0;
let coordinates = []; // Will store as {lat, lng} for Leaflet
let coordinatesUTM = []; // Will store as {easting, northing} for display
let calculatedArea = 0;
let mapInitialized = false;

// ============================================
// VALIDATION FUNCTIONS (match edit-tab-pengukuran)
// ============================================

// Coordinate validation function for UTM
function isValidUTMCoordinate(easting, northing) {
    return (
        typeof easting === 'number' &&
        typeof northing === 'number' &&
        !isNaN(easting) &&
        !isNaN(northing) &&
        easting > 0 &&
        northing > 0
    );
}

// Original lat/lng validation (still needed for map operations)
function isValidCoordinate(lat, lng) {
    return (
        typeof lat === 'number' &&
        typeof lng === 'number' &&
        !isNaN(lat) &&
        !isNaN(lng) &&
        lat >= -90 &&
        lat <= 90 &&
        lng >= -180 &&
        lng <= 180
    );
}

// Validate coordinate input on blur
function validateCoordinateInput(input) {
    if (!input || !input.value) return true;

    const value = parseFloat(input.value);
    if (isNaN(value) || value <= 0) {
        input.style.borderColor = '#dc3545';
        input.title = 'Harus berupa angka positif';
        return false;
    }
    input.style.borderColor = '';
    input.title = '';
    return true;
}

// Update coordinate from UTM input change (EXACT copy from edit-tab-pengukuran)
function updateFromUTM(index) {
    const eastingInput = document.getElementById(`easting_${index}`);
    const northingInput = document.getElementById(`northing_${index}`);

    if (!eastingInput || !northingInput) {
        console.warn('UTM inputs not found for index:', index);
        return;
    }

    const easting = parseFloat(eastingInput.value);
    const northing = parseFloat(northingInput.value);

    if (isNaN(easting) || isNaN(northing)) {
        console.warn('Invalid UTM values');
        return;
    }

    // Validate UTM coordinates
    if (!validateCoordinateInput(eastingInput) || !validateCoordinateInput(northingInput)) {
        console.warn('UTM validation failed');
        return;
    }

    // Convert UTM to Lat/Lng
    const latLng = utmToLatLng(easting, northing);

    if (!latLng) {
        console.error('Failed to convert UTM to Lat/Lng');
        return;
    }

    console.log(`UTM (${easting}, ${northing}) → Lat/Lng (${latLng.lat}, ${latLng.lng})`);

    // Update the coordinates array (Leaflet needs Lat/Lng)
    const coordIndex = coordinates.findIndex((c, i) => {
        const row = document.getElementById(`coord-${index}`);
        return row && Array.from(document.querySelectorAll('#coordinatesList tr')).indexOf(row) === i;
    });

    if (coordIndex >= 0) {
        coordinates[coordIndex] = latLng;
    }

    // Update map and calculations
    updateMapAndCalculate();
}

// Initialize map with Leaflet layers (100% match edit-tab-pengukuran logic)
function initMap() {
    if (mapInitialized && map) {
        console.log('Map already initialized, updating...');
        updateMapPolygon();
        return;
    }

    try {
        // Clear any existing map instance
        if (map) {
            map.remove();
            map = null;
        }

        // Ensure map container exists
        const mapContainer = document.getElementById('polygonMap');
        if (!mapContainer) {
            console.error('Map container not found');
            return;
        }

        // Check internet connection
        if (!navigator.onLine) {
            alert('Tidak ada koneksi internet');
            return;
        }

        // Initialize map with bounds checking
        const center = coordinates.length > 0
            ? [coordinates[0].lat, coordinates[0].lng]
            : [-1.2417, 116.8730]; // Default Berau coordinates
        const zoom = 18;

        // Validate coordinates before initialization
        if (!isValidCoordinate(center[0], center[1])) {
            throw new Error('Invalid center coordinates');
        }

        map = L.map('polygonMap', {
            center: center,
            zoom: zoom,
            maxZoom: 21,  // Increased to 21 for very close zoom
            minZoom: 1,
            zoomControl: true,
            attributionControl: true,
            preferCanvas: false
        });

        // Primary tile layer with error handling
        const osmLayer = L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '© OpenStreetMap contributors',
            maxZoom: 21,  // Increased to 21
            minZoom: 1,
            errorTileUrl: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==',
            tileSize: 256,
            subdomains: ['a', 'b', 'c']
        });

        // Alternative tile layers as fallbacks
        const cartoDB = L.tileLayer('https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}{r}.png', {
            attribution: '© CARTO © OpenStreetMap contributors',
            maxZoom: 21,  // Increased to 21
            minZoom: 1,
            subdomains: 'abcd'
        });

        // Handle tile errors and auto-fallback
        let tileErrorCount = 0;
        osmLayer.on('tileerror', function(error) {
            tileErrorCount++;
            console.warn('OSM tile error:', tileErrorCount);

            if (tileErrorCount >= 3 && !map.hasLayer(cartoDB)) {
                console.log('Switching to CartoDB due to tile errors');
                map.removeLayer(osmLayer);
                cartoDB.addTo(map);
            }
        });

        // Add primary layer
        osmLayer.addTo(map);

        // Layer control
        const baseLayers = {
            "OpenStreetMap": osmLayer,
            "CartoDB Light": cartoDB
        };

        L.control.layers(baseLayers).addTo(map);

        // Add scale control
        L.control.scale({
            position: 'bottomright',
            metric: true,
            imperial: false
        }).addTo(map);

        // Map click event handler
        map.on('click', function(e) {
            const lat = e.latlng.lat;
            const lng = e.latlng.lng;

            if (!isValidCoordinate(lat, lng)) {
                alert('Koordinat yang diklik tidak valid!');
                return;
            }

            const latRounded = lat.toFixed(6);
            const lngRounded = lng.toFixed(6);

            // Convert to UTM for display
            const utmCoord = latLngToUTM(lat, lng);

            if (!utmCoord) {
                alert('Gagal mengkonversi koordinat ke UTM');
                return;
            }

            if (confirm(`Tambah koordinat?\nEasting: ${utmCoord.easting.toFixed(2)} m\nNorthing: ${utmCoord.northing.toFixed(2)} m`)) {
                addCoordinateWithValues(lat, lng);
            }
        });

        mapInitialized = true;
        console.log('Map initialized successfully');

        // Load existing coordinates into table
        loadExistingCoordinates();

        // Initial update with delay
        setTimeout(function() {
            updateMapAndCalculate();
        }, 100);

    } catch (error) {
        console.error('Error initializing map:', error);
        alert('Gagal memuat peta. Silakan coba lagi atau periksa koneksi internet.');
        mapInitialized = false;
    }
}

// Update map and calculate area (EXACT copy from edit-tab-pengukuran)
function updateMapAndCalculate() {
    if (!map) {
        console.warn('Map not initialized, skipping update');
        return;
    }

    try {
        console.log('updateMapAndCalculate called');

        // Clear existing markers and polygon
        markers.forEach(marker => {
            try {
                if (map.hasLayer(marker)) {
                    map.removeLayer(marker);
                }
            } catch (e) {
                console.warn('Error removing marker:', e);
            }
        });
        markers = [];

        if (polygon && map.hasLayer(polygon)) {
            try {
                map.removeLayer(polygon);
            } catch (e) {
                console.warn('Error removing polygon:', e);
            }
        }

        // Get coordinates from UTM inputs with validation
        coordinates = [];
        coordinatesUTM = [];
        const rows = document.querySelectorAll('#coordinatesList tr');

        console.log('Found rows:', rows.length);

        rows.forEach((row, index) => {
            const coordId = row.id ? row.id.split('-')[1] : (index + 1);
            const eastingInput = document.getElementById(`easting_${coordId}`);
            const northingInput = document.getElementById(`northing_${coordId}`);

            if (eastingInput && northingInput && eastingInput.value && northingInput.value) {
                const easting = parseFloat(eastingInput.value);
                const northing = parseFloat(northingInput.value);

                console.log(`Row ${index + 1}: Easting=${easting}, Northing=${northing}`);

                // Reset border colors
                eastingInput.style.borderColor = '';
                northingInput.style.borderColor = '';

                // Validate UTM coordinates
                if (isValidUTMCoordinate(easting, northing)) {
                    // Store UTM coordinates
                    coordinatesUTM.push({easting: easting, northing: northing});

                    // Convert UTM to Lat/Lng for map display
                    const latLng = utmToLatLng(easting, northing);

                    console.log(`Row ${index + 1}: UTM (${easting}, ${northing}) → Lat/Lng (${latLng?.lat.toFixed(10)}, ${latLng?.lng.toFixed(10)})`);

                    if (latLng && isValidCoordinate(latLng.lat, latLng.lng)) {
                        coordinates.push(latLng);
                        console.log(`  → ✓ Valid coordinate added to array`);

                        // Add numbered marker
                        try {
                            const marker = L.marker([latLng.lat, latLng.lng], {
                                icon: L.divIcon({
                                    className: 'custom-div-icon',
                                    html: `<div style="background: #28a745; color: white; width: 28px; height: 28px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 12px; font-weight: bold; border: 2px solid white; box-shadow: 0 2px 6px rgba(0,0,0,0.3);">${index + 1}</div>`,
                                    iconSize: [28, 28],
                                    iconAnchor: [14, 14]
                                })
                            });

                            marker.addTo(map);
                            markers.push(marker);

                            // Add popup with both UTM and Lat/Lng
                            marker.bindPopup(`
                                <strong>Titik ${index + 1}</strong><br>
                                <b>UTM Zone 50N:</b><br>
                                Easting: ${easting.toFixed(2)} m<br>
                                Northing: ${northing.toFixed(2)} m<br>
                                <hr style="margin: 5px 0;">
                                <b>Lat/Lng:</b><br>
                                Lat: ${latLng.lat.toFixed(8)}°<br>
                                Lng: ${latLng.lng.toFixed(8)}°
                            `);

                        } catch (markerError) {
                            console.error('Error adding marker:', markerError);
                        }
                    } else {
                        console.warn(`UTM conversion failed at index ${index}:`, easting, northing);
                        eastingInput.style.borderColor = '#dc3545';
                        northingInput.style.borderColor = '#dc3545';
                    }
                } else {
                    console.warn(`Invalid UTM coordinate at index ${index}:`, easting, northing);
                    eastingInput.style.borderColor = '#dc3545';
                    northingInput.style.borderColor = '#dc3545';
                }
            }
        });

        // Update coordinate count
        const countElement = document.getElementById('coordinateCount');
        if (countElement) {
            countElement.textContent = coordinates.length;
        }
        console.log('Total valid coordinates:', coordinates.length);

        // Draw polygon if we have at least 3 valid coordinates
        if (coordinates.length >= 3) {
            try {
                const latlngs = coordinates.map(coord => [coord.lat, coord.lng]);
                console.log('=== POLYGON DEBUG ===');
                console.log('Total coordinates:', coordinates.length);
                console.log('Coordinate objects:', coordinates);
                console.log('Latlngs array for polygon:', latlngs);

                // Check if coordinates are actually different
                if (coordinates.length >= 3) {
                    const dx1 = coordinates[1].lng - coordinates[0].lng;
                    const dy1 = coordinates[1].lat - coordinates[0].lat;
                    const dx2 = coordinates[2].lng - coordinates[0].lng;
                    const dy2 = coordinates[2].lat - coordinates[0].lat;
                    const crossProduct = dx1 * dy2 - dy1 * dx2;
                    console.log('Cross product (should be != 0 for triangle):', crossProduct);
                    if (Math.abs(crossProduct) < 1e-10) {
                        console.error('⚠️  WARNING: Points are collinear (forming a line, not triangle)!');
                    } else {
                        console.log('✓ Points form a valid triangle');
                    }
                }

                polygon = L.polygon(latlngs, {
                    color: '#28a745',
                    fillColor: '#28a745',
                    fillOpacity: 0.3,
                    weight: 3,
                    dashArray: '5, 5'
                });

                polygon.addTo(map);

                console.log('✓ Polygon drawn successfully with', coordinates.length, 'points');

                // Calculate and display area
                calculateArea();

                // Add polygon popup
                polygon.bindPopup(`
                    <strong>Area Polygon</strong><br>
                    Luas: ${document.getElementById('calculatedArea').textContent}<br>
                    Titik: ${coordinates.length}
                `);

                // Update hidden inputs
                updateHiddenInputs();

                // Fit map to polygon bounds (EXACT copy from edit-tab-pengukuran)
                const bounds = polygon.getBounds();
                if (bounds.isValid()) {
                    map.fitBounds(bounds, {padding: [50, 50]});
                }                // Auto-capture map whenever valid polygon exists (delay to let map render)
                setTimeout(() => {
                    captureMap();
                }, 500);
            } catch (polygonError) {
                console.error('Error creating polygon:', polygonError);
            }
        } else {
            console.warn('Not enough coordinates to draw polygon:', coordinates.length);
            // Reset area if insufficient coordinates
            document.getElementById('calculatedArea').textContent = '0 m²';
            // Clear map_image_data if polygon is invalid
            document.getElementById('map_image_data').value = '';
            updateHiddenInputs();
        }

    } catch (error) {
        console.error('Error updating polygon:', error);
    }
}

// Update hidden inputs with coordinate data
function updateHiddenInputs() {
    try {
        // Update coordinates_data with lat/lng for backward compatibility
        document.getElementById('coordinates_data').value = JSON.stringify(coordinates);

        // Create/update hidden inputs for lat/lng (for database storage)
        const hiddenContainer = document.getElementById('latLngHiddenInputs');
        hiddenContainer.innerHTML = '';

        coordinates.forEach((coord, index) => {
            // Add lat hidden input
            const latInput = document.createElement('input');
            latInput.type = 'hidden';
            latInput.name = `lat[${index + 1}]`;
            latInput.value = coord.lat;
            hiddenContainer.appendChild(latInput);

            // Add lng hidden input
            const lngInput = document.createElement('input');
            lngInput.type = 'hidden';
            lngInput.name = `lng[${index + 1}]`;
            lngInput.value = coord.lng;
            hiddenContainer.appendChild(lngInput);
        });
    } catch (error) {
        console.error('Error updating hidden inputs:', error);
    }
}

// Add new coordinate row (100% match edit-tab-pengukuran.blade.php)
function addCoordinate() {
    coordinateCount++;
    const tbody = document.getElementById('coordinatesList');
    const rowNumber = tbody.children.length + 1;

    const newRow = document.createElement('tr');
    newRow.id = `coord-${coordinateCount}`;
    newRow.innerHTML = `
        <td class="align-middle font-weight-bold">${rowNumber}</td>
        <td>
            <input type="text" class="form-control form-control-sm"
                   name="easting[${coordinateCount}]" id="easting_${coordinateCount}"
                   placeholder="Easting (m)"
                   onchange="updateFromUTM(${coordinateCount})"
                   onblur="validateCoordinateInput(this)">
        </td>
        <td>
            <input type="text" class="form-control form-control-sm"
                   name="northing[${coordinateCount}]" id="northing_${coordinateCount}"
                   placeholder="Northing (m)"
                   onchange="updateFromUTM(${coordinateCount})"
                   onblur="validateCoordinateInput(this)">
        </td>
        <td class="text-center">
            <button type="button" class="btn btn-danger btn-sm"
                    onclick="removeCoordinate(${coordinateCount})" title="Hapus">
                <i class="fas fa-times"></i>
            </button>
        </td>
    `;
    tbody.appendChild(newRow);
}

// Add coordinate with values (from map click) - 100% match edit-tab-pengukuran.blade.php
function addCoordinateWithValues(lat, lng) {
    coordinateCount++;
    const tbody = document.getElementById('coordinatesList');
    const rowNumber = tbody.children.length + 1;

    // Convert Lat/Lng to UTM for display
    const utmCoord = latLngToUTM(lat, lng);

    if (!utmCoord) {
        alert('Gagal mengkonversi koordinat ke UTM');
        return;
    }

    const newRow = document.createElement('tr');
    newRow.id = `coord-${coordinateCount}`;
    newRow.innerHTML = `
        <td class="align-middle font-weight-bold">${rowNumber}</td>
        <td>
            <input type="text" class="form-control form-control-sm"
                   name="easting[${coordinateCount}]" id="easting_${coordinateCount}"
                   value="${utmCoord.easting.toFixed(2)}"
                   onchange="updateFromUTM(${coordinateCount})"
                   onblur="validateCoordinateInput(this)">
        </td>
        <td>
            <input type="text" class="form-control form-control-sm"
                   name="northing[${coordinateCount}]" id="northing_${coordinateCount}"
                   value="${utmCoord.northing.toFixed(2)}"
                   onchange="updateFromUTM(${coordinateCount})"
                   onblur="validateCoordinateInput(this)">
        </td>
        <td class="text-center">
            <button type="button" class="btn btn-danger btn-sm"
                    onclick="removeCoordinate(${coordinateCount})" title="Hapus">
                <i class="fas fa-times"></i>
            </button>
        </td>
    `;
    tbody.appendChild(newRow);

    // Update arrays and map
    coordinates.push({lat: lat, lng: lng});
    coordinatesUTM.push({easting: utmCoord.easting, northing: utmCoord.northing});
    updateMapAndCalculate();
}

// Remove coordinate row
function removeCoordinate(id) {
    if (confirm('Hapus koordinat ini?')) {
        const row = document.getElementById(`coord-${id}`);
        if (row) {
            row.remove();

            // Re-number rows
            const rows = document.querySelectorAll('#coordinatesList tr');
            rows.forEach((row, index) => {
                row.querySelector('td:first-child').textContent = index + 1;
            });

            updateMapAndCalculate();
        }
    }
}

// Calculate polygon area using UTM coordinates (MORE ACCURATE)
function calculatePolygonArea() {
    if (coordinatesUTM.length < 3) return 0;

    try {
        let area = 0;
        const n = coordinatesUTM.length;

        // Shoelace formula using UTM coordinates (in meters)
        for (let i = 0; i < n; i++) {
            const j = (i + 1) % n;
            area += coordinatesUTM[i].easting * coordinatesUTM[j].northing;
            area -= coordinatesUTM[j].easting * coordinatesUTM[i].northing;
        }

        area = Math.abs(area / 2);

        console.log('Calculated area (UTM-based):', area.toFixed(2), 'm²');

        return Math.round(area * 100) / 100;
    } catch (error) {
        console.error('Error calculating area:', error);
        return 0;
    }
}

// Calculate and display area
function calculateArea() {
    if (coordinates.length < 3) {
        calculatedArea = 0;
        document.getElementById('calculatedArea').textContent = '0 m²';
        document.getElementById('luas_terukur').value = '0';
        return 0;
    }

    try {
        // Use calculatePolygonArea function
        const areaInSquareMeters = calculatePolygonArea();

        // Update global variable
        calculatedArea = areaInSquareMeters;

        // Update display
        document.getElementById('calculatedArea').textContent =
            areaInSquareMeters.toLocaleString('id-ID', {maximumFractionDigits: 2}) + ' m²';

        // Save to hidden input for backend
        document.getElementById('luas_terukur').value = areaInSquareMeters.toFixed(2);

        return areaInSquareMeters;
    } catch (error) {
        console.error('Error calculating area:', error);
        calculatedArea = 0;
        document.getElementById('calculatedArea').textContent = 'Error';
        document.getElementById('luas_terukur').value = '0';
        return 0;
    }
}

// Center map on polygon
function centerMap() {
    try {
        if (polygon) {
            map.fitBounds(polygon.getBounds());
        } else if (coordinates.length > 0) {
            map.setView([coordinates[0].lat, coordinates[0].lng], 15);
        } else {
            map.setView([-1.2379, 116.8529], 13);
        }
    } catch (error) {
        console.error('Error centering map:', error);
    }
}

// Capture map as image
function captureMap(showAlert = false) {
    try {
        if (coordinates.length < 3) {
            if (showAlert) {
                alert('Minimal 3 koordinat diperlukan untuk capture peta');
            }
            return false;
        }

        // Show loading indicator
        if (showAlert) {
            Swal.fire({
                title: 'Sedang capture peta...',
                text: 'Mohon tunggu sebentar',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });
        }

        // Get polygon bounds
        const bounds = polygon.getBounds();
        const sw = bounds.getSouthWest(); // Southwest corner
        const ne = bounds.getNorthEast(); // Northeast corner

        // Add padding (8% on each side) - lebih kecil untuk zoom lebih dekat
        const latRange = ne.lat - sw.lat;
        const lngRange = ne.lng - sw.lng;
        const latPadding = latRange * 0.08;
        const lngPadding = lngRange * 0.08;

        const minLat = sw.lat - latPadding;
        const maxLat = ne.lat + latPadding;
        const minLng = sw.lng - lngPadding;
        const maxLng = ne.lng + lngPadding;

        // Calculate aspect ratio to maintain polygon shape
        const totalLatRange = maxLat - minLat;
        const totalLngRange = maxLng - minLng;
        const aspectRatio = totalLngRange / totalLatRange;

        // Create canvas with dynamic size based on polygon shape - diperbesar ke 1200px
        const canvas = document.createElement('canvas');
        let canvasWidth, canvasHeight;
        const maxDimension = 1200; // Diperbesar dari 800 ke 1200 untuk polygon lebih besar

        // Set max dimension, adjust other dimension to maintain aspect ratio
        if (aspectRatio > 1) {
            // Landscape (wider than tall)
            canvasWidth = maxDimension;
            canvasHeight = Math.round(maxDimension / aspectRatio);
        } else {
            // Portrait (taller than wide)
            canvasHeight = maxDimension;
            canvasWidth = Math.round(maxDimension * aspectRatio);
        }

        canvas.width = canvasWidth;
        canvas.height = canvasHeight;
        const ctx = canvas.getContext('2d');

        // White background
        ctx.fillStyle = '#ffffff';
        ctx.fillRect(0, 0, canvasWidth, canvasHeight);

        // Convert lat/lng to canvas coordinates (using proper map projection)
        function latLngToCanvas(lat, lng) {
            const x = ((lng - minLng) / (maxLng - minLng)) * canvasWidth;
            const y = ((maxLat - lat) / (maxLat - minLat)) * canvasHeight;
            return { x, y };
        }

        // Get polygon latlngs from Leaflet (this ensures exact same coordinates as displayed)
        const polygonLatLngs = polygon.getLatLngs()[0];

        // Draw polygon
        ctx.beginPath();
        polygonLatLngs.forEach((latlng, index) => {
            const point = latLngToCanvas(latlng.lat, latlng.lng);
            if (index === 0) {
                ctx.moveTo(point.x, point.y);
            } else {
                ctx.lineTo(point.x, point.y);
            }
        });
        ctx.closePath();

        // Fill polygon with same color as Leaflet (yellow with opacity)
        ctx.fillStyle = 'rgba(255, 255, 0, 0.3)'; // #ffff00 with 0.3 opacity
        ctx.fill();

        // Stroke polygon with same color as Leaflet (red)
        ctx.strokeStyle = '#ff0000';
        ctx.lineWidth = 4; // Diperbesar dari 3 ke 4
        ctx.stroke();

        // Draw coordinate points and labels - diperbesar dan lebih jelas
        polygonLatLngs.forEach((latlng, index) => {
            const point = latLngToCanvas(latlng.lat, latlng.lng);

            // Draw point circle - diperbesar
            ctx.beginPath();
            ctx.arc(point.x, point.y, 10, 0, 2 * Math.PI); // Diperbesar dari 6 ke 10
            ctx.fillStyle = '#ff0000';
            ctx.fill();
            ctx.strokeStyle = '#ffffff';
            ctx.lineWidth = 3; // Diperbesar dari 2 ke 3
            ctx.stroke();

            // Draw point label with background - diperbesar
            const label = (index + 1).toString();
            ctx.font = 'bold 20px Arial'; // Diperbesar dari 14px ke 20px
            const textMetrics = ctx.measureText(label);
            const textWidth = textMetrics.width;
            const textHeight = 24; // Disesuaikan dengan font size

            // Background for label - lebih besar dan lebih jelas
            ctx.fillStyle = 'rgba(255, 255, 255, 0.95)'; // Lebih opaque
            ctx.fillRect(point.x + 14, point.y - textHeight, textWidth + 12, textHeight + 6);

            // Border untuk background label
            ctx.strokeStyle = '#000000';
            ctx.lineWidth = 1;
            ctx.strokeRect(point.x + 14, point.y - textHeight, textWidth + 12, textHeight + 6);

            // Label text
            ctx.fillStyle = '#000000';
            ctx.fillText(label, point.x + 20, point.y - 6);
        });

        // Convert canvas to compressed JPEG base64 image
        const imageData = canvas.toDataURL('image/jpeg', 0.7); // JPEG, quality 0.7

        // Store in hidden input
        document.getElementById('map_image_data').value = imageData;

        console.log('Polygon captured successfully, image data length:', imageData.length);

        if (showAlert) {
            Swal.fire({
                icon: 'success',
                title: 'Berhasil!',
                text: 'Polygon berhasil di-capture',
                timer: 2000,
                showConfirmButton: false
            });
        }

        return true;

    } catch (error) {
        console.error('Error in captureMap function:', error);

        if (showAlert) {
            Swal.fire({
                icon: 'error',
                title: 'Gagal!',
                text: 'Gagal capture polygon. Silakan coba lagi.',
            });
        }

        return false;
    }
}

// Load existing coordinates if editing (100% match edit-tab-pengukuran.blade.php)
function loadExistingCoordinates() {
    try {
        const tbody = document.getElementById('coordinatesList');
        if (!tbody) {
            console.error('coordinatesList element not found');
            return;
        }

        tbody.innerHTML = '';
        coordinateCount = 0;
        coordinates = [];
        coordinatesUTM = [];

        // Use existing coordinates or create default ones (as Lat/Lng, will convert to UTM)
        let coordsToLoad;

        @if(isset($spSipt) && $spSipt->coordinates)
            const existingCoords = @json($spSipt->coordinates_array);
            coordsToLoad = existingCoords.map(coord => {
                const utm = latLngToUTM(coord.lat, coord.lng);
                return {
                    lat: coord.lat,
                    lng: coord.lng,
                    easting: utm ? utm.easting.toFixed(2) : '',
                    northing: utm ? utm.northing.toFixed(2) : ''
                };
            });
        @else
            // Default Berau coordinates (same as edit-tab-pengukuran)
            coordsToLoad = [
                {lat: -1.2417, lng: 116.8730},
                {lat: -1.2418, lng: 116.8729},
                {lat: -1.2419, lng: 116.8731}
            ].map(coord => {
                const utm = latLngToUTM(coord.lat, coord.lng);
                console.log(`Default coord: Lat=${coord.lat}, Lng=${coord.lng} → UTM E=${utm ? utm.easting : 'null'}, N=${utm ? utm.northing : 'null'}`);
                return {
                    lat: coord.lat,
                    lng: coord.lng,
                    easting: utm ? utm.easting.toFixed(2) : '',
                    northing: utm ? utm.northing.toFixed(2) : ''
                };
            });
        @endif

        coordsToLoad.forEach((coord, index) => {
            // Validate using lat/lng if available, otherwise convert from UTM
            let lat = coord.lat;
            let lng = coord.lng;
            let easting = coord.easting || '';
            let northing = coord.northing || '';

            // If we don't have lat/lng but have UTM, convert
            if ((!lat || !lng) && easting && northing) {
                const latLng = utmToLatLng(easting, northing);
                if (latLng) {
                    lat = latLng.lat;
                    lng = latLng.lng;
                }
            }

            // If we don't have UTM but have lat/lng, convert
            if ((!easting || !northing) && lat && lng) {
                const utm = latLngToUTM(lat, lng);
                if (utm) {
                    easting = utm.easting.toFixed(2);
                    northing = utm.northing.toFixed(2);
                }
            }

            if (!isValidCoordinate(lat, lng)) {
                console.warn('Invalid coordinate at index', index, ':', coord);
                return;
            }

            coordinateCount++;
            // CRITICAL: Push to arrays so updateMapAndCalculate() has data!
            coordinates.push({lat: lat, lng: lng});
            coordinatesUTM.push({easting: parseFloat(easting), northing: parseFloat(northing)});

            console.log(`Loaded coord ${coordinateCount}: Lat=${lat}, Lng=${lng}, E=${easting}, N=${northing}`);

            const row = document.createElement('tr');
            row.id = `coord-${coordinateCount}`;
            row.innerHTML = `
                <td class="align-middle font-weight-bold">${index + 1}</td>
                <td>
                    <input type="text"
                           class="form-control form-control-sm"
                           id="easting_${coordinateCount}"
                           name="easting[${coordinateCount}]"
                           value="${easting}"
                           onchange="updateFromUTM(${coordinateCount})"
                           onblur="validateCoordinateInput(this)"
                           placeholder="Easting (m)">
                </td>
                <td>
                    <input type="text"
                           class="form-control form-control-sm"
                           id="northing_${coordinateCount}"
                           name="northing[${coordinateCount}]"
                           value="${northing}"
                           onchange="updateFromUTM(${coordinateCount})"
                           onblur="validateCoordinateInput(this)"
                           placeholder="Northing (m)">
                </td>
                <td class="text-center">
                    <button type="button" onclick="removeCoordinate(${coordinateCount})"
                            class="btn btn-danger btn-sm" title="Hapus koordinat">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            `;

            tbody.appendChild(row);
        });

        console.log('Loaded coordinates:', coordinateCount, 'points (UTM format)');
    } catch (error) {
        console.error('Error loading existing coordinates:', error);
    }
}

// File upload handler
function showFileInfo(input, previewId) {
    try {
        const file = input.files[0];
        if (file) {
            const fileName = file.name;
            const fileSize = (file.size / 1024 / 1024).toFixed(2);

            $('#' + previewId).html(`
                <div class="alert alert-success">
                    <i class="fas fa-file-pdf mr-2"></i>
                    <strong>${fileName}</strong> (${fileSize} MB)
                    <button type="button" class="close" onclick="clearFile('${input.id}', '${previewId}')">
                        <span>&times;</span>
                    </button>
                </div>
            `);
        }
    } catch (error) {
        console.error('Error handling file upload:', error);
    }
}

// Clear file input
function clearFile(inputId, previewId) {
    try {
        $('#' + inputId).val('');
        $('#' + previewId).html('');
    } catch (error) {
        console.error('Error clearing file input:', error);
    }
}

// Form validation
function initializeValidation() {
    try {
        // Initialize Select2 for dropdowns
        $('.select2').select2();

        // NIK validation
        $('input[name="pemohon_nik"], input[name="terlapor_nik"]').on('input', function() {
            this.value = this.value.replace(/[^0-9]/g, '').substring(0, 16);
        });

        // Form submit validation
        $('#formSpSipt').on('submit', function(e) {
            let isValid = true;
            const errors = [];

            // Check required fields
            const requiredFields = {
                'pemohon_nama': 'Nama Pihak Pertama',
                'pemohon_nik': 'NIK Pihak Pertama',
                'pemohon_alamat': 'Alamat Pihak Pertama',
                'terlapor_nama': 'Nama Pihak Kedua',
                'kampung': 'Kampung/Kelurahan',
                'kecamatan_id': 'Kecamatan'
            };

            $.each(requiredFields, function(field, label) {
                const value = $(`[name="${field}"]`).val();
                if (!value || value.trim() === '') {
                    errors.push(`${label} harus diisi`);
                    $(`[name="${field}"]`).addClass('is-invalid');
                    isValid = false;
                }
            });

            // Check NIK length
            const nik = $('input[name="pemohon_nik"]').val();
            if (nik && nik.length !== 16) {
                errors.push('NIK Pihak Pertama harus 16 digit');
                $('input[name="pemohon_nik"]').addClass('is-invalid');
                isValid = false;
            }

            const terlaporNik = $('input[name="terlapor_nik"]').val();
            if (terlaporNik && terlaporNik.length !== 16) {
                errors.push('NIK Pihak Kedua harus 16 digit');
                $('input[name="terlapor_nik"]').addClass('is-invalid');
                isValid = false;
            }

            // Check coordinates
            if (coordinates.length < 3) {
                errors.push('Minimal 3 koordinat diperlukan untuk membentuk polygon');
                isValid = false;
            }

            // Check map_image_data
            const mapImageData = $('#map_image_data').val();
            if (!mapImageData || mapImageData.trim() === '') {
                errors.push('Data gambar peta belum ter-capture. Silakan tunggu sebentar atau klik tombol Capture manual.');
                isValid = false;

                // Try to auto-capture one more time
                if (coordinates.length >= 3) {
                    captureMap();
                }
            }

            if (!isValid) {
                e.preventDefault();

                // Show errors
                if (errors.length > 0) {
                    let errorMsg = '<ul class="mb-0">';
                    errors.forEach(error => {
                        errorMsg += `<li>${error}</li>`;
                    });
                    errorMsg += '</ul>';

                    $('#errorModal').remove();
                    $('body').append(`
                        <div class="modal fade" id="errorModal" tabindex="-1">
                            <div class="modal-dialog">
                                <div class="modal-content">
                                    <div class="modal-header bg-danger text-white">
                                        <h5 class="modal-title">Validasi Gagal</h5>
                                        <button type="button" class="close text-white" data-dismiss="modal">
                                            <span>&times;</span>
                                        </button>
                                    </div>
                                    <div class="modal-body">
                                        ${errorMsg}
                                    </div>
                                    <div class="modal-footer">
                                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Tutup</button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    `);
                    $('#errorModal').modal('show');
                }

                // Scroll to first invalid field
                const firstInvalid = $('.is-invalid').first();
                if (firstInvalid.length) {
                    $('html, body').animate({
                        scrollTop: firstInvalid.offset().top - 100
                    }, 500);
                }
            }
        });

        // Remove invalid class on input
        $('input, textarea, select').on('input change', function() {
            $(this).removeClass('is-invalid');
        });
    } catch (error) {
        console.error('Error initializing validation:', error);
    }
}

// Utility functions
function scrollToBottom() {
    try {
        $('html, body').animate({
            scrollTop: $(document).height()
        }, 1000);
    } catch (error) {
        console.error('Error scrolling to bottom:', error);
    }
}

// Initialize everything when DOM is ready
$(document).ready(function() {
    try {
        console.log('Document ready - Create mode');

        // Initialize map - this will also call loadExistingCoordinates() and updateMapAndCalculate()
        initMap();

        // Initialize validation
        initializeValidation();

        // Initialize tooltips
        $('[data-toggle="tooltip"]').tooltip();
    } catch (error) {
        console.error('Error during initialization:', error);
        alert('Terjadi kesalahan saat memuat halaman. Silakan coba lagi.');
    }
});
</script>
@endpush
