@extends('admin-layouts.master')

@section('title', 'Create New User')

@section('content')
<div class="container-fluid">
    <!-- Header Section -->
    <div class="row mb-4">
        <div class="col-md-12">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h2 class="mb-1 text-dark font-weight-bold">Create New User</h2>
                    <p class="text-muted mb-0">Add a new user to the system with appropriate roles</p>
                </div>
                <div>
                    <a href="{{ route('users.index') }}" class="btn btn-light btn-sm shadow-sm">
                        <i class="fas fa-arrow-left mr-1"></i> Back to Users
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Alert Messages -->
    @if($errors->any())
    <div class="row mb-3">
        <div class="col-md-12">
            <div class="alert alert-danger alert-dismissible fade show shadow-sm border-0" role="alert">
                <i class="fas fa-exclamation-circle mr-2"></i>
                <strong>Please fix the following errors:</strong>
                <ul class="mb-0 mt-2">
                    @foreach($errors->all() as $error)
                    <li>{{ $error }}</li>
                    @endforeach
                </ul>
                <button type="button" class="close" data-dismiss="alert">
                    <span>&times;</span>
                </button>
            </div>
        </div>
    </div>
    @endif

    <form action="{{ route('users.store') }}" method="POST" enctype="multipart/form-data" id="userForm">
        @csrf
        <div class="row">
            <!-- User Information Card -->
            <div class="col-md-8 mb-4">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-white border-bottom-0 py-3">
                        <h6 class="mb-0 text-dark font-weight-semibold">
                            <i class="fas fa-user text-primary mr-2"></i>
                            User Information
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <!-- Username -->
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="username" class="form-label font-weight-semibold">Username <span class="text-danger">*</span></label>
                                    <input type="text"
                                           class="form-control @error('username') is-invalid @enderror"
                                           id="username"
                                           name="username"
                                           value="{{ old('username') }}"
                                           placeholder="Enter username"
                                           required>
                                    <small class="form-text text-muted">Unique identifier for login</small>
                                    @error('username')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <!-- Full Name -->
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="name" class="form-label font-weight-semibold">Full Name <span class="text-danger">*</span></label>
                                    <input type="text"
                                           class="form-control @error('name') is-invalid @enderror"
                                           id="name"
                                           name="name"
                                           value="{{ old('name') }}"
                                           placeholder="Enter full name"
                                           required>
                                    @error('name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Email -->
                        <div class="form-group">
                            <label for="email" class="form-label font-weight-semibold">Email Address <span class="text-danger">*</span></label>
                            <input type="email"
                                   class="form-control @error('email') is-invalid @enderror"
                                   id="email"
                                   name="email"
                                   value="{{ old('email') }}"
                                   placeholder="Enter email address"
                                   required>
                            <small class="form-text text-muted">Will be used for login and notifications</small>
                            @error('email')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="row">
                            <!-- Password -->
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="password" class="form-label font-weight-semibold">Password <span class="text-danger">*</span></label>
                                    <div class="input-group">
                                        <input type="password"
                                               class="form-control @error('password') is-invalid @enderror"
                                               id="password"
                                               name="password"
                                               placeholder="Enter password"
                                               required>
                                        <div class="input-group-append">
                                            <button class="btn btn-outline-secondary" type="button" id="togglePassword">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                        </div>
                                        @error('password')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    <small class="form-text text-muted">Minimum 8 characters</small>
                                </div>
                            </div>

                            <!-- Confirm Password -->
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="password_confirmation" class="form-label font-weight-semibold">Confirm Password <span class="text-danger">*</span></label>
                                    <div class="input-group">
                                        <input type="password"
                                               class="form-control"
                                               id="password_confirmation"
                                               name="password_confirmation"
                                               placeholder="Confirm password"
                                               required>
                                        <div class="input-group-append">
                                            <button class="btn btn-outline-secondary" type="button" id="togglePasswordConfirm">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Password Strength Indicator -->
                        <div class="form-group">
                            <div class="progress" style="height: 5px;">
                                <div class="progress-bar" id="passwordStrength" role="progressbar" style="width: 0%"></div>
                            </div>
                            <small class="form-text" id="passwordStrengthText">Password strength will appear here</small>
                        </div>

                        <!-- Profile Photo -->
                        <div class="form-group">
                            <label for="profile_photo" class="form-label font-weight-semibold">Profile Photo</label>
                            <div class="custom-file">
                                <input type="file"
                                       class="custom-file-input @error('profile_photo') is-invalid @enderror"
                                       id="profile_photo"
                                       name="profile_photo"
                                       accept="image/jpeg,image/png,image/jpg">
                                <label class="custom-file-label" for="profile_photo">Choose file</label>
                                @error('profile_photo')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <small class="form-text text-muted">Max 2MB (JPEG, PNG, JPG)</small>

                            <!-- Image Preview -->
                            <div id="imagePreview" class="mt-3" style="display: none;">
                                <img id="preview" src="" alt="Preview" class="img-thumbnail" style="max-width: 200px;">
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Roles Assignment Card -->
            <div class="col-md-4 mb-4">
                <div class="card border-0 shadow-sm h-100">
                    <div class="card-header bg-white border-bottom-0 py-3">
                        <h6 class="mb-0 text-dark font-weight-semibold">
                            <i class="fas fa-user-tag text-primary mr-2"></i>
                            Assign Roles
                        </h6>
                    </div>
                    <div class="card-body">
                        @if($roles->count() > 0)
                        <!-- Primary Role -->
                        <div class="form-group">
                            <label for="primary_role_id" class="form-label font-weight-semibold">Primary Role <span class="text-danger">*</span></label>
                            <select class="form-control @error('primary_role_id') is-invalid @enderror"
                                    id="primary_role_id"
                                    name="primary_role_id"
                                    required>
                                <option value="">Select Primary Role</option>
                                @foreach($roles as $role)
                                <option value="{{ $role->id }}"
                                        {{ old('primary_role_id') == $role->id ? 'selected' : '' }}
                                        data-description="{{ $role->description }}">
                                    {{ $role->name }}
                                </option>
                                @endforeach
                            </select>
                            <small class="form-text text-muted">Main role for this user</small>
                            @error('primary_role_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Role Description -->
                        <div id="roleDescription" class="mb-3 p-3 bg-light rounded" style="display: none;">
                            <small class="text-muted"></small>
                        </div>

                        <!-- Additional Roles -->
                        <div class="form-group">
                            <label class="form-label font-weight-semibold">Additional Roles</label>
                            <small class="form-text text-muted mb-2">Select all roles for this user (including primary)</small>

                            @foreach($roles as $role)
                            <div class="custom-control custom-checkbox mb-2">
                                <input type="checkbox"
                                       class="custom-control-input role-checkbox"
                                       id="role{{ $role->id }}"
                                       name="roles[]"
                                       value="{{ $role->id }}"
                                       {{ in_array($role->id, old('roles', [])) ? 'checked' : '' }}>
                                <label class="custom-control-label" for="role{{ $role->id }}">
                                    <div class="d-flex align-items-center">
                                        <div class="role-icon mr-2">
                                            @php
                                                $roleConfig = match($role->name) {
                                                    'superadmin' => ['bg' => 'bg-danger', 'icon' => 'fa-crown'],
                                                    'supervisor' => ['bg' => 'bg-warning', 'icon' => 'fa-user-tie'],
                                                    'staff' => ['bg' => 'bg-info', 'icon' => 'fa-user-edit'],
                                                    'administrator' => ['bg' => 'bg-success', 'icon' => 'fa-user-cog'],
                                                    'guest' => ['bg' => 'bg-secondary', 'icon' => 'fa-user-clock'],
                                                    default => ['bg' => 'bg-secondary', 'icon' => 'fa-user']
                                                };
                                            @endphp
                                            <div class="{{ $roleConfig['bg'] }} rounded-circle d-flex align-items-center justify-content-center" style="width: 25px; height: 25px;">
                                                <i class="fas {{ $roleConfig['icon'] }} text-white" style="font-size: 10px;"></i>
                                            </div>
                                        </div>
                                        <div>
                                            <span class="font-weight-medium">{{ $role->display_name }}</span>
                                            @if($role->description)
                                            <br><small class="text-muted">{{ Str::limit($role->description, 30) }}</small>
                                            @endif
                                        </div>
                                    </div>
                                </label>
                            </div>
                            @endforeach
                            @error('roles')
                                <div class="text-danger small">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Selected Roles Summary -->
                        <div class="mt-3 p-3 bg-light rounded">
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <span class="font-weight-semibold text-dark">Selected Roles</span>
                                <span class="badge badge-primary" id="selectedRolesCount">0</span>
                            </div>
                            <div id="selectedRolesList" class="text-muted small">
                                No roles selected
                            </div>
                        </div>

                        @else
                        <div class="text-center py-4">
                            <i class="fas fa-exclamation-triangle fa-2x text-warning mb-2"></i>
                            <h6 class="text-muted">No Roles Available</h6>
                            <p class="text-muted small">Please create roles first before adding users.</p>
                            @can('role.create')
                            <a href="{{ route('roles.create') }}" class="btn btn-sm btn-primary">
                                <i class="fas fa-plus mr-1"></i> Create Role
                            </a>
                            @endcan
                        </div>
                        @endif
                    </div>
                </div>
            </div>
        </div>

        <!-- Action Buttons -->
        <div class="row">
            <div class="col-md-12">
                <div class="card border-0 shadow-sm">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <a href="{{ route('users.index') }}" class="btn btn-light px-4">
                                <i class="fas fa-times mr-1"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-primary px-4" id="submitBtn">
                                <i class="fas fa-save mr-1"></i> Create User
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </form>
</div>

@push('styles')
<style>
.card {
    border-radius: 8px;
    transition: all 0.3s ease;
}

.form-control {
    border-radius: 6px;
    border: 1px solid #e3e6f0;
}

.form-control:focus {
    border-color: #4e73df;
    box-shadow: 0 0 0 0.2rem rgba(78, 115, 223, 0.25);
}

.custom-control-label {
    cursor: pointer;
}

.custom-control-input:checked ~ .custom-control-label::before {
    background-color: #4e73df;
    border-color: #4e73df;
}

.role-icon {
    flex-shrink: 0;
}

.badge {
    font-size: 0.75rem;
    font-weight: 500;
}

.form-label {
    font-size: 0.85rem;
    margin-bottom: 0.25rem;
}

.alert {
    border-radius: 6px;
}

.input-group-text {
    border-radius: 0 6px 6px 0;
}

.input-group .form-control {
    border-radius: 6px 0 0 6px;
}

.progress-bar {
    transition: width 0.3s ease;
}

.progress-bar.bg-danger {
    background-color: #dc3545 !important;
}

.progress-bar.bg-warning {
    background-color: #ffc107 !important;
}

.progress-bar.bg-info {
    background-color: #17a2b8 !important;
}

.progress-bar.bg-success {
    background-color: #28a745 !important;
}

.role-item {
    background-color: #e7f1ff;
    color: #004085;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 0.75rem;
    margin: 2px;
    display: inline-block;
}

.custom-file-label::after {
    content: "Browse";
}
</style>
@endpush

@push('scripts')
<script>
$(document).ready(function() {
    // Password visibility toggle
    $('#togglePassword').click(function() {
        const password = $('#password');
        const type = password.attr('type') === 'password' ? 'text' : 'password';
        password.attr('type', type);
        $(this).find('i').toggleClass('fa-eye fa-eye-slash');
    });

    $('#togglePasswordConfirm').click(function() {
        const password = $('#password_confirmation');
        const type = password.attr('type') === 'password' ? 'text' : 'password';
        password.attr('type', type);
        $(this).find('i').toggleClass('fa-eye fa-eye-slash');
    });

    // Profile photo preview
    $('#profile_photo').change(function() {
        const file = this.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
                $('#preview').attr('src', e.target.result);
                $('#imagePreview').show();
            }
            reader.readAsDataURL(file);

            // Update file label
            $(this).next('.custom-file-label').text(file.name);
        }
    });

    // Password strength checker
    $('#password').on('input', function() {
        const password = $(this).val();
        const strength = checkPasswordStrength(password);

        $('#passwordStrength')
            .removeClass('bg-danger bg-warning bg-info bg-success')
            .addClass(strength.class)
            .css('width', strength.percentage + '%');

        $('#passwordStrengthText')
            .text(strength.text)
            .removeClass('text-danger text-warning text-info text-success')
            .addClass(strength.textClass);
    });

    function checkPasswordStrength(password) {
        let score = 0;

        if (password.length >= 8) score++;
        if (password.match(/[a-z]/)) score++;
        if (password.match(/[A-Z]/)) score++;
        if (password.match(/[0-9]/)) score++;
        if (password.match(/[^a-zA-Z0-9]/)) score++;

        switch(score) {
            case 0:
            case 1:
                return { percentage: 20, class: 'bg-danger', text: 'Very Weak', textClass: 'text-danger' };
            case 2:
                return { percentage: 40, class: 'bg-warning', text: 'Weak', textClass: 'text-warning' };
            case 3:
                return { percentage: 60, class: 'bg-info', text: 'Fair', textClass: 'text-info' };
            case 4:
                return { percentage: 80, class: 'bg-success', text: 'Good', textClass: 'text-success' };
            case 5:
                return { percentage: 100, class: 'bg-success', text: 'Very Strong', textClass: 'text-success' };
        }
    }

    // Role selection handling
    function updateSelectedRoles() {
        const checked = $('.role-checkbox:checked');
        const count = checked.length;
        $('#selectedRolesCount').text(count);

        if (count === 0) {
            $('#selectedRolesList').html('<span class="text-muted">No roles selected</span>');
        } else {
            let html = '';
            checked.each(function() {
                const label = $(this).next('label').find('span').first().text();
                html += '<span class="role-item">' + label + '</span>';
            });
            $('#selectedRolesList').html(html);
        }
    }

    // Primary role change handler
    $('#primary_role_id').change(function() {
        const selectedValue = $(this).val();
        const description = $(this).find('option:selected').data('description');

        if (selectedValue) {
            $('.role-checkbox').prop('checked', false);
            $(`#role${selectedValue}`).prop('checked', true);

            if (description) {
                $('#roleDescription small').text(description);
                $('#roleDescription').show();
            } else {
                $('#roleDescription').hide();
            }
        } else {
            $('#roleDescription').hide();
        }

        updateSelectedRoles();
    });

    $('.role-checkbox').change(function() {
        updateSelectedRoles();

        const primaryRoleId = $('#primary_role_id').val();
        if (!$(this).is(':checked') && $(this).val() === primaryRoleId) {
            $('#primary_role_id').val('');
            $('#roleDescription').hide();
        }
    });

    // Form validation
    $('#userForm').submit(function(e) {
        const selectedRoles = $('.role-checkbox:checked').length;
        const primaryRole = $('#primary_role_id').val();

        if (selectedRoles === 0) {
            e.preventDefault();
            alert('Please select at least one role for this user.');
            return false;
        }

        if (!primaryRole) {
            e.preventDefault();
            alert('Please select a primary role for this user.');
            return false;
        }

        if (!$(`#role${primaryRole}`).is(':checked')) {
            e.preventDefault();
            alert('Primary role must be one of the selected roles.');
            return false;
        }

        const password = $('#password').val();
        const passwordConfirm = $('#password_confirmation').val();

        if (password !== passwordConfirm) {
            e.preventDefault();
            alert('Password and confirm password do not match.');
            return false;
        }

        $('#submitBtn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-1"></i> Creating...');
    });

    updateSelectedRoles();

    setTimeout(function() {
        $('.alert').fadeOut('slow');
    }, 5000);
});
</script>
@endpush
@endsection
