{{-- File: resources/views/components/quick-booking-modal.blade.php - PART 1 --}}

<style>
/* Quick Booking Modal Styles */
.quick-booking-modal .modal-dialog {
    max-width: 900px;
}

.quick-booking-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 25px 30px;
    border-radius: 12px 12px 0 0;
    position: relative;
}

.quick-booking-header::after {
    content: '';
    position: absolute;
    bottom: -10px;
    left: 50%;
    transform: translateX(-50%);
    width: 0;
    height: 0;
    border-left: 15px solid transparent;
    border-right: 15px solid transparent;
    border-top: 15px solid #764ba2;
}

.quick-booking-body {
    padding: 30px;
    background: #f8f9fa;
    max-height: 75vh;
    overflow-y: auto;
}

/* Form Sections */
.form-section {
    background: white;
    border-radius: 15px;
    padding: 25px;
    margin-bottom: 25px;
    border: 2px solid #e9ecef;
    box-shadow: 0 4px 15px rgba(0,0,0,0.08);
    transition: all 0.3s ease;
}

.form-section:hover {
    border-color: #667eea;
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.15);
}

.section-title {
    font-size: 16px;
    font-weight: 700;
    color: #2d3436;
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
    padding-bottom: 10px;
    border-bottom: 2px solid #f1f3f4;
}

.section-title i {
    width: 20px;
    text-align: center;
}

/* Form Controls */
.quick-booking-modal .form-control {
    border: 2px solid #e9ecef;
    border-radius: 10px;
    padding: 12px 15px;
    font-size: 14px;
    transition: all 0.3s ease;
    background: white;
}

.quick-booking-modal .form-control:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 0.25rem rgba(102, 126, 234, 0.15);
    background: white;
}

.quick-booking-modal .form-label {
    font-weight: 600;
    color: #495057;
    margin-bottom: 8px;
    font-size: 13px;
}

/* Guest Toggle */
.guest-toggle {
    display: flex;
    background: #e9ecef;
    border-radius: 10px;
    padding: 5px;
    margin-bottom: 20px;
    box-shadow: inset 0 2px 4px rgba(0,0,0,0.1);
}

.guest-toggle-btn {
    flex: 1;
    background: none;
    border: none;
    padding: 10px 15px;
    border-radius: 7px;
    font-size: 14px;
    font-weight: 600;
    transition: all 0.3s ease;
    color: #6c757d;
    cursor: pointer;
}

.guest-toggle-btn.active {
    background: white;
    color: #667eea;
    box-shadow: 0 3px 8px rgba(0,0,0,0.15);
    transform: translateY(-1px);
}

.guest-toggle-btn:hover:not(.active) {
    color: #495057;
}

/* Guest Selection */
.guest-select-container {
    max-height: 200px;
    overflow-y: auto;
    border: 2px solid #e9ecef;
    border-radius: 10px;
    background: white;
}

.guest-option {
    padding: 15px;
    border-bottom: 1px solid #f1f3f4;
    cursor: pointer;
    transition: all 0.2s ease;
    position: relative;
}

.guest-option:hover {
    background: #f8f9fa;
    transform: translateX(5px);
}

.guest-option.selected {
    background: linear-gradient(135deg, #e3f2fd, #f0f8ff);
    color: #1976d2;
    border-left: 4px solid #667eea;
}

.guest-option:last-child {
    border-bottom: none;
}

.guest-name {
    font-weight: 600;
    font-size: 15px;
    margin-bottom: 5px;
    color: #2d3436;
}

.guest-option.selected .guest-name {
    color: #1976d2;
}

.guest-info {
    font-size: 13px;
    color: #6c757d;
    line-height: 1.4;
}

.guest-info i {
    width: 15px;
    text-align: center;
    margin-right: 5px;
}

/* Check Icon */
.guest-check-icon {
    position: absolute;
    top: 15px;
    right: 15px;
    color: #28a745;
    font-size: 16px;
    opacity: 0;
    transform: scale(0);
    transition: all 0.3s ease;
}

.guest-option.selected .guest-check-icon {
    opacity: 1;
    transform: scale(1);
}

/* Date Section */
.date-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
    margin-bottom: 20px;
}

@media (max-width: 576px) {
    .date-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }
}

.date-info {
    background: linear-gradient(135deg, #e3f2fd, #f8f9ff);
    border: 2px solid #667eea;
    border-radius: 12px;
    padding: 20px;
    text-align: center;
    margin-top: 15px;
    animation: slideUp 0.3s ease;
}

@keyframes slideUp {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

.nights-count {
    font-size: 28px;
    font-weight: bold;
    color: #667eea;
    margin-bottom: 5px;
}

.nights-text {
    color: #495057;
    font-weight: 500;
}

/* Room Section */
.room-search-container {
    position: relative;
    margin-bottom: 20px;
}

.room-search-icon {
    position: absolute;
    left: 15px;
    top: 50%;
    transform: translateY(-50%);
    color: #6c757d;
    z-index: 2;
}

.room-search-input {
    padding-left: 45px;
}

.rooms-container {
    max-height: 300px;
    overflow-y: auto;
    padding-right: 5px;
}

/* Custom Scrollbar */
.rooms-container::-webkit-scrollbar,
.guest-select-container::-webkit-scrollbar {
    width: 6px;
}

.rooms-container::-webkit-scrollbar-track,
.guest-select-container::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 3px;
}

.rooms-container::-webkit-scrollbar-thumb,
.guest-select-container::-webkit-scrollbar-thumb {
    background: #c1c1c1;
    border-radius: 3px;
}

.rooms-container::-webkit-scrollbar-thumb:hover,
.guest-select-container::-webkit-scrollbar-thumb:hover {
    background: #a8a8a8;
}

/* Loading States */
.loading-state {
    text-align: center;
    padding: 40px 20px;
    color: #6c757d;
}

.loading-state i {
    font-size: 40px;
    margin-bottom: 15px;
    color: #667eea;
}

.spinner {
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.empty-state {
    text-align: center;
    padding: 40px 20px;
    color: #6c757d;
}

.empty-state i {
    font-size: 48px;
    color: #dee2e6;
    margin-bottom: 15px;
}

/* Responsive Design */
@media (max-width: 768px) {
    .quick-booking-modal .modal-dialog {
        margin: 10px;
        max-width: calc(100% - 20px);
    }
    
    .quick-booking-body {
        padding: 20px;
        max-height: 80vh;
    }
    
    .form-section {
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .section-title {
        font-size: 15px;
    }
}

@media (max-width: 576px) {
    .quick-booking-header {
        padding: 20px;
        text-align: center;
    }
    
    .quick-booking-body {
        padding: 15px;
    }
    
    .form-section {
        padding: 15px;
    }
}
</style>

<!-- Quick Booking Modal -->
<div class="modal fade quick-booking-modal" id="quickBookingModal" tabindex="-1" aria-labelledby="quickBookingModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <!-- Header -->
            <div class="quick-booking-header">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h4 class="modal-title mb-1" id="quickBookingModalLabel">
                            <i class="fas fa-calendar-check me-2"></i>Quick Booking
                        </h4>
                        <p class="mb-0 opacity-85" style="font-size: 14px;">Buat reservasi kamar dengan cepat dan mudah</p>
                    </div>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
            </div>

            <!-- Body -->
            <div class="quick-booking-body">
                <form id="quickBookingForm">
                    @csrf
                    
                    <!-- Guest Section -->
                    <div class="form-section">
                        <div class="section-title">
                            <i class="fas fa-user text-primary"></i>
                            Data Tamu
                        </div>
                        
                        <div class="guest-toggle">
                            <button type="button" class="guest-toggle-btn active" id="existingGuestBtn">
                                <i class="fas fa-users me-2"></i>Pilih Tamu
                            </button>
                            <button type="button" class="guest-toggle-btn" id="newGuestBtn">
                                <i class="fas fa-user-plus me-2"></i>Tamu Baru
                            </button>
                        </div>

                        <!-- Existing Guest Section -->
                        <div id="existingGuestSection">
                            <div class="mb-3">
                                <input type="text" class="form-control" id="guestSearch" placeholder="🔍 Cari nama, telepon, atau email tamu...">
                            </div>
                            <div class="guest-select-container" id="guestList">
                                <div class="loading-state">
                                    <i class="fas fa-spinner spinner"></i>
                                    <div>Memuat data tamu...</div>
                                </div>
                            </div>
                        </div>

                        <!-- New Guest Section -->
                        <div id="newGuestSection" style="display: none;">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Nama Lengkap *</label>
                                    <input type="text" class="form-control" id="newGuestName" placeholder="Masukkan nama lengkap" required>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">No. Telepon</label>
                                    <input type="text" class="form-control" id="newGuestPhone" placeholder="Contoh: 081234567890">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Email</label>
                                    <input type="email" class="form-control" id="newGuestEmail" placeholder="contoh@email.com">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">No. Identitas</label>
                                    <input type="text" class="form-control" id="newGuestIdentity" placeholder="KTP/Passport">
                                </div>
                                <div class="col-12">
                                    <label class="form-label">Alamat</label>
                                    <textarea class="form-control" id="newGuestAddress" rows="2" placeholder="Alamat lengkap (opsional)"></textarea>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Date Section -->
                    <div class="form-section">
                        <div class="section-title">
                            <i class="fas fa-calendar text-info"></i>
                            Tanggal Menginap
                        </div>
                        
                        <div class="date-grid">
                            <div>
                                <label class="form-label">
                                    <i class="fas fa-sign-in-alt text-success me-1"></i>Check-in
                                </label>
                                <input type="date" class="form-control" id="checkInDate" required>
                            </div>
                            <div>
                                <label class="form-label">
                                    <i class="fas fa-sign-out-alt text-danger me-1"></i>Check-out
                                </label>
                                <input type="date" class="form-control" id="checkOutDate" required>
                            </div>
                        </div>

                        <div class="date-info" id="dateInfo" style="display: none;">
                            <div class="nights-count" id="nightsCount">0</div>
                            <div class="nights-text">malam menginap</div>
                        </div>
                    </div>

                    <!-- Room Section akan di Part 2 -->
                </form>
            </div>
        </div>
    </div>
</div>
{{-- File: resources/views/components/quick-booking-modal.blade.php - PART 2 --}}
{{-- Lanjutan dari Part 1, tambahkan setelah Date Section --}}

                    <!-- Room Section -->
                    <div class="form-section">
                        <div class="section-title">
                            <i class="fas fa-bed text-success"></i>
                            Pilih Kamar Tersedia
                        </div>
                        
                        <div class="room-search-container">
                            <i class="fas fa-search room-search-icon"></i>
                            <input type="text" class="form-control room-search-input" id="searchRoomKeyword" placeholder="Cari tipe kamar atau nomor kamar...">
                        </div>
                        
                        <div class="rooms-container" id="roomsList">
                            <div class="loading-state">
                                <i class="fas fa-bed"></i>
                                <div>Pilih tanggal untuk melihat kamar tersedia</div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Footer -->
            <div class="modal-footer bg-light">
                <button type="button" class="btn btn-secondary-quick btn-quick" data-bs-dismiss="modal">
                    <i class="fas fa-times me-2"></i>Batal
                </button>
                <button type="button" class="btn btn-success-quick btn-quick" id="confirmBookingBtn" disabled>
                    <i class="fas fa-check me-2"></i>Booking Sekarang
                </button>
            </div>
        </div>
    </div>
</div>

<style>
/* Room Card Styles */
.room-card {
    border: 2px solid #e9ecef;
    border-radius: 15px;
    padding: 20px;
    margin-bottom: 15px;
    cursor: pointer;
    transition: all 0.3s ease;
    background: white;
    position: relative;
    overflow: hidden;
}

.room-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 5px;
    height: 100%;
    background: linear-gradient(180deg, #28a745, #20c997);
    transform: scaleY(0);
    transition: transform 0.3s ease;
    transform-origin: bottom;
}

.room-card:hover {
    border-color: #28a745;
    transform: translateY(-3px);
    box-shadow: 0 10px 30px rgba(40, 167, 69, 0.2);
}

.room-card:hover::before {
    transform: scaleY(1);
}

.room-card.selected {
    border-color: #28a745;
    background: linear-gradient(135deg, #f8fff8, #e8f5e8);
    box-shadow: 0 10px 30px rgba(40, 167, 69, 0.25);
    transform: translateY(-2px);
}

.room-card.selected::before {
    transform: scaleY(1);
}

.room-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 15px;
}

.room-details h6 {
    margin: 0 0 8px 0;
    font-size: 16px;
    font-weight: 700;
    color: #2d3436;
}

.room-card.selected .room-details h6 {
    color: #28a745;
}

.room-description {
    color: #6c757d;
    font-size: 13px;
    margin-bottom: 8px;
    line-height: 1.4;
}

.room-capacity {
    color: #495057;
    font-size: 13px;
    font-weight: 500;
}

.room-capacity i {
    color: #6c757d;
    margin-right: 5px;
}

.room-pricing {
    text-align: right;
    min-width: 120px;
}

.room-price {
    font-size: 18px;
    font-weight: bold;
    color: #28a745;
    margin-bottom: 3px;
}

.room-card.selected .room-price {
    color: #20c997;
}

.room-per-night {
    font-size: 11px;
    color: #6c757d;
    line-height: 1.2;
}

.room-check-icon {
    position: absolute;
    top: 15px;
    right: 15px;
    color: #28a745;
    font-size: 20px;
    opacity: 0;
    transform: scale(0);
    transition: all 0.3s ease;
}

.room-card.selected .room-check-icon {
    opacity: 1;
    transform: scale(1);
}

/* Button Styles */
.btn-quick {
    border-radius: 10px;
    padding: 12px 25px;
    font-size: 14px;
    font-weight: 600;
    transition: all 0.3s ease;
    border: none;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 140px;
}

.btn-quick:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 20px rgba(0, 0, 0, 0.15);
}

.btn-secondary-quick {
    background: #6c757d;
    color: white;
}

.btn-secondary-quick:hover {
    background: #545b62;
    color: white;
}

.btn-success-quick {
    background: linear-gradient(135deg, #28a745, #20c997);
    color: white;
    position: relative;
    overflow: hidden;
}

.btn-success-quick::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
    transition: left 0.5s;
}

.btn-success-quick:hover::before {
    left: 100%;
}

.btn-success-quick:hover {
    background: linear-gradient(135deg, #20c997, #17a2b8);
    color: white;
    box-shadow: 0 8px 25px rgba(40, 167, 69, 0.4);
}

.btn-success-quick:disabled {
    background: #dee2e6;
    color: #6c757d;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

.btn-success-quick:disabled:hover {
    background: #dee2e6;
    color: #6c757d;
    transform: none;
    box-shadow: none;
}

/* Modal Footer */
.modal-footer {
    border-top: 2px solid #e9ecef;
    padding: 20px 30px;
    background: #f8f9fa;
}

/* Success Animations */
@keyframes bounceIn {
    0% { 
        transform: scale(0.3); 
        opacity: 0; 
    }
    50% { 
        transform: scale(1.05); 
        opacity: 0.8;
    }
    70% { 
        transform: scale(0.9); 
        opacity: 0.9;
    }
    100% { 
        transform: scale(1); 
        opacity: 1; 
    }
}

.bounce-in {
    animation: bounceIn 0.6s ease-out;
}

/* Pulse Animation for Selected Items */
@keyframes pulse {
    0% { box-shadow: 0 0 0 0 rgba(40, 167, 69, 0.4); }
    70% { box-shadow: 0 0 0 10px rgba(40, 167, 69, 0); }
    100% { box-shadow: 0 0 0 0 rgba(40, 167, 69, 0); }
}

.room-card.selected {
    animation: pulse 2s infinite;
}

.guest-option.selected {
    animation: pulse 2s infinite;
}

/* Fade In Animation for Dynamic Content */
@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.fade-in-up {
    animation: fadeInUp 0.5s ease-out;
}

/* Enhanced Focus States */
.quick-booking-modal .form-control:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 0.25rem rgba(102, 126, 234, 0.15);
    transform: translateY(-1px);
}

/* Improved Scrollbar for Better UX */
.quick-booking-body::-webkit-scrollbar {
    width: 8px;
}

.quick-booking-body::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 4px;
}

.quick-booking-body::-webkit-scrollbar-thumb {
    background: linear-gradient(180deg, #667eea, #764ba2);
    border-radius: 4px;
}

.quick-booking-body::-webkit-scrollbar-thumb:hover {
    background: linear-gradient(180deg, #5a67d8, #6b46c1);
}

/* Mobile Optimizations */
@media (max-width: 576px) {
    .room-header {
        flex-direction: column;
        gap: 10px;
    }
    
    .room-pricing {
        text-align: left;
        min-width: auto;
    }
    
    .btn-quick {
        min-width: auto;
        width: 100%;
        margin-bottom: 10px;
    }
    
    .modal-footer {
        flex-direction: column;
        gap: 10px;
    }
    
    .modal-footer .btn:last-child {
        margin-bottom: 0;
    }
}

/* Dark Theme Support (Optional) */
@media (prefers-color-scheme: dark) {
    .form-section {
        background: #2d3748;
        border-color: #4a5568;
        color: #e2e8f0;
    }
    
    .quick-booking-modal .form-control {
        background: #4a5568;
        border-color: #68d391;
        color: #e2e8f0;
    }
    
    .guest-option {
        border-color: #4a5568;
    }
    
    .room-card {
        background: #2d3748;
        border-color: #4a5568;
        color: #e2e8f0;
    }
}

/* Print Styles */
@media print {
    .quick-booking-modal {
        display: none !important;
    }
}

/* High Contrast Mode Support */
@media (prefers-contrast: high) {
    .form-section {
        border-width: 3px;
    }
    
    .guest-option.selected,
    .room-card.selected {
        border-width: 4px;
    }
    
    .btn-quick {
        border: 2px solid currentColor;
    }
}

/* Reduced Motion Support */
@media (prefers-reduced-motion: reduce) {
    .room-card,
    .guest-option,
    .btn-quick,
    .form-section {
        transition: none;
    }
    
    .spinner {
        animation: none;
    }
    
    .bounce-in,
    .fade-in-up {
        animation: none;
    }
}
</style>
@push('scripts')
    
<script>
    // Quick Booking Modal JavaScript - Part 3 (Guest & Date Management)

$(document).ready(function() {
    // Global variables
    let selectedGuest = null;
    let selectedRoom = null;
    let availableRooms = [];

    // Initialize Quick Booking
    initializeQuickBooking();

    function initializeQuickBooking() {
        // Set minimum date to today
        const today = new Date().toISOString().split('T')[0];
        $('#checkInDate').attr('min', today).val(today);
        $('#checkOutDate').attr('min', today);

        // Auto-set checkout to tomorrow
        const tomorrow = new Date();
        tomorrow.setDate(tomorrow.getDate() + 1);
        $('#checkOutDate').val(tomorrow.toISOString().split('T')[0]);
        
        // Calculate initial nights
        calculateNights();

        // Setup event listeners
        setupEventListeners();
        
        // Load initial data when modal opens
        $('#quickBookingModal').on('shown.bs.modal', function() {
            loadGuests();
            $('#guestSearch').focus();
        });

        // Reset form when modal closes
        $('#quickBookingModal').on('hidden.bs.modal', function() {
            resetForm();
        });
    }

    function setupEventListeners() {
        // Guest toggle buttons
        $('.guest-toggle-btn').click(function() {
            $('.guest-toggle-btn').removeClass('active');
            $(this).addClass('active');
            
            if ($(this).attr('id') === 'existingGuestBtn') {
                $('#existingGuestSection').show();
                $('#newGuestSection').hide();
                selectedGuest = null;
                $('.guest-option').removeClass('selected');
            } else {
                $('#existingGuestSection').hide();
                $('#newGuestSection').show();
                selectedGuest = null;
                $('#newGuestName').focus();
            }
            updateBookingButton();
        });

        // Guest search with debounce
        let searchTimer;
        $('#guestSearch').on('input', function() {
            clearTimeout(searchTimer);
            const keyword = $(this).val();
            
            searchTimer = setTimeout(() => {
                searchGuests(keyword);
            }, 400);
        });

        // New guest form handling
        $('#newGuestName, #newGuestPhone, #newGuestEmail, #newGuestIdentity, #newGuestAddress').on('input', function() {
            if ($('#newGuestName').val().trim()) {
                selectedGuest = {
                    name: $('#newGuestName').val().trim(),
                    phone: $('#newGuestPhone').val().trim(),
                    email: $('#newGuestEmail').val().trim(),
                    identity: $('#newGuestIdentity').val().trim(),
                    address: $('#newGuestAddress').val().trim(),
                    type: 'new'
                };
            } else {
                selectedGuest = null;
            }
            updateBookingButton();
        });

        // Date change handlers
        $('#checkInDate').change(function() {
            const checkInDate = new Date($(this).val());
            checkInDate.setDate(checkInDate.getDate() + 1);
            const minCheckOut = checkInDate.toISOString().split('T')[0];
            $('#checkOutDate').attr('min', minCheckOut);
            
            // Auto-adjust checkout if it's before new minimum
            if ($('#checkOutDate').val() <= $(this).val()) {
                $('#checkOutDate').val(minCheckOut);
            }
            
            calculateNights();
            if ($('#checkOutDate').val()) {
                loadAvailableRooms();
            }
        });

        $('#checkOutDate').change(function() {
            calculateNights();
            if ($('#checkInDate').val()) {
                loadAvailableRooms();
            }
        });

        // Room search with debounce
        let roomSearchTimer;
        $('#searchRoomKeyword').on('input', function() {
            clearTimeout(roomSearchTimer);
            roomSearchTimer = setTimeout(() => {
                if ($('#checkInDate').val() && $('#checkOutDate').val()) {
                    loadAvailableRooms();
                }
            }, 400);
        });

        // Confirm booking button
        $('#confirmBookingBtn').click(confirmBooking);
    }

    // Guest Management Functions
    function loadGuests(keyword = '') {
        $('#guestList').html(`
            <div class="loading-state">
                <i class="fas fa-spinner spinner"></i>
                <div>Memuat data tamu...</div>
            </div>
        `);

        $.ajax({
            url: '{{ route("api.guests.search") }}',
            method: 'GET',
            data: { search: keyword, limit: 20 },
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            },
            success: function(response) {
                if (response.success) {
                    displayGuests(response.data);
                } else {
                    showGuestError('Gagal memuat data tamu');
                }
            },
            error: function(xhr) {
                console.error('Guest load error:', xhr);
                showGuestError('Terjadi kesalahan saat memuat data tamu');
            }
        });
    }

    function searchGuests(keyword) {
        if (keyword.length === 0) {
            loadGuests(); // Load default guests
        } else if (keyword.length >= 2) {
            loadGuests(keyword);
        }
    }

    function displayGuests(guests) {
        let html = '';
        
        if (guests.length === 0) {
            html = `
                <div class="empty-state">
                    <i class="fas fa-user-slash"></i>
                    <div>Tidak ada tamu ditemukan</div>
                </div>
            `;
        } else {
            guests.forEach(guest => {
                html += `
                    <div class="guest-option fade-in-up" data-guest-id="${guest.id}">
                        <div class="guest-name">${escapeHtml(guest.name)}</div>
                        <div class="guest-info">
                            <i class="fas fa-phone"></i>${guest.phone || 'Tidak ada telepon'}
                            <br>
                            <i class="fas fa-envelope"></i>${guest.email || 'Tidak ada email'}
                        </div>
                        <i class="fas fa-check-circle guest-check-icon"></i>
                    </div>
                `;
            });
        }

        $('#guestList').html(html);

        // Add click handlers for guest selection
        $('.guest-option').click(function() {
            selectGuest($(this), guests);
        });
    }

    function selectGuest(element, guests) {
        // Remove previous selection
        $('.guest-option').removeClass('selected');
        element.addClass('selected bounce-in');
        
        const guestId = element.data('guest-id');
        const guest = guests.find(g => g.id == guestId);
        
        selectedGuest = {
            id: guest.id,
            name: guest.name,
            phone: guest.phone,
            email: guest.email,
            type: 'existing'
        };
        
        // Clear new guest form
        clearNewGuestForm();
        
        // Scroll to selected guest
        element[0].scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        
        updateBookingButton();
        
        // Show success feedback
        showToast('Tamu dipilih: ' + guest.name, 'success');
    }

    function clearNewGuestForm() {
        $('#newGuestName, #newGuestPhone, #newGuestEmail, #newGuestIdentity, #newGuestAddress').val('');
    }

    function showGuestError(message) {
        $('#guestList').html(`
            <div class="empty-state">
                <i class="fas fa-exclamation-triangle text-warning"></i>
                <div>${message}</div>
            </div>
        `);
    }

    // Date Management Functions
    function calculateNights() {
        const checkIn = new Date($('#checkInDate').val());
        const checkOut = new Date($('#checkOutDate').val());
        
        if (checkIn && checkOut && checkOut > checkIn) {
            const nights = Math.ceil((checkOut - checkIn) / (1000 * 60 * 60 * 24));
            $('#nightsCount').text(nights);
            $('#dateInfo').show().addClass('fade-in-up');
            return nights;
        } else {
            $('#dateInfo').hide().removeClass('fade-in-up');
            return 0;
        }
    }

    function validateDates() {
        const checkIn = $('#checkInDate').val();
        const checkOut = $('#checkOutDate').val();
        
        if (!checkIn || !checkOut) {
            return { valid: false, message: 'Silakan pilih tanggal check-in dan check-out' };
        }
        
        if (new Date(checkIn) >= new Date(checkOut)) {
            return { valid: false, message: 'Tanggal check-out harus setelah check-in' };
        }
        
        if (new Date(checkIn) < new Date().setHours(0,0,0,0)) {
            return { valid: false, message: 'Tanggal check-in tidak boleh di masa lalu' };
        }
        
        return { valid: true };
    }

    // Utility Functions
    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }

    function showToast(message, type = 'info') {
        // Simple toast notification
        const toast = $(`
            <div class="toast-notification toast-${type}" style="
                position: fixed; 
                top: 20px; 
                right: 20px; 
                z-index: 9999;
                background: ${type === 'success' ? '#28a745' : type === 'error' ? '#dc3545' : '#17a2b8'};
                color: white;
                padding: 12px 20px;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                transform: translateX(100%);
                transition: transform 0.3s ease;
            ">
                <i class="fas fa-${type === 'success' ? 'check' : type === 'error' ? 'times' : 'info'} me-2"></i>
                ${message}
            </div>
        `);
        
        $('body').append(toast);
        
        // Animate in
        setTimeout(() => toast.css('transform', 'translateX(0)'), 100);
        
        // Auto remove
        setTimeout(() => {
            toast.css('transform', 'translateX(100%)');
            setTimeout(() => toast.remove(), 300);
        }, 3000);
    }

    function updateBookingButton() {
        const hasGuest = selectedGuest !== null;
        const hasRoom = selectedRoom !== null;
        const datesValid = validateDates().valid;
        
        const isValid = hasGuest && hasRoom && datesValid;
        $('#confirmBookingBtn').prop('disabled', !isValid);
        
        // Update button text based on validation
        let buttonText = '<i class="fas fa-check me-2"></i>Booking Sekarang';
        if (!hasGuest) {
            buttonText = '<i class="fas fa-user me-2"></i>Pilih Tamu Dulu';
        } else if (!datesValid) {
            buttonText = '<i class="fas fa-calendar me-2"></i>Periksa Tanggal';
        } else if (!hasRoom) {
            buttonText = '<i class="fas fa-bed me-2"></i>Pilih Kamar Dulu';
        }
        
        $('#confirmBookingBtn').html(buttonText);
    }

    function resetForm() {
        selectedGuest = null;
        selectedRoom = null;
        availableRooms = [];
        
        // Reset form elements
        $('#quickBookingForm')[0].reset();
        $('#guestSearch').val('');
        $('#searchRoomKeyword').val('');
        
        // Reset UI states
        $('.guest-option, .room-card').removeClass('selected');
        $('#existingGuestBtn').click();
        $('#dateInfo').hide();
        $('#roomsList').html(`
            <div class="loading-state">
                <i class="fas fa-bed"></i>
                <div>Pilih tanggal untuk melihat kamar tersedia</div>
            </div>
        `);
        
        // Reset dates
        const today = new Date().toISOString().split('T')[0];
        $('#checkInDate').val(today);
        
        const tomorrow = new Date();
        tomorrow.setDate(tomorrow.getDate() + 1);
        $('#checkOutDate').val(tomorrow.toISOString().split('T')[0]);
        
        calculateNights();
        updateBookingButton();
    }

    // Expose functions globally for access from other parts
    window.quickBookingFunctions = {
        loadGuests,
        searchGuests,
        calculateNights,
        validateDates,
        updateBookingButton,
        resetForm,
        showToast
    };
});
// Quick Booking Modal JavaScript - Part 4 (Room Management & Booking)

// Room Management Functions (add to existing script)
function loadAvailableRooms() {
    const dateValidation = validateDates();
    if (!dateValidation.valid) {
        $('#roomsList').html(`
            <div class="empty-state">
                <i class="fas fa-calendar-times text-warning"></i>
                <div>${dateValidation.message}</div>
            </div>
        `);
        return;
    }

    $('#roomsList').html(`
        <div class="loading-state">
            <i class="fas fa-spinner spinner"></i>
            <div>Mencari kamar tersedia...</div>
        </div>
    `);

    $.ajax({
        url: '{{ route("api.bookings.check-availability") }}',
        method: 'POST',
        data: {
            check_in_date: $('#checkInDate').val(),
            check_out_date: $('#checkOutDate').val(),
            search: $('#searchRoomKeyword').val()
        },
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        },
        success: function(response) {
            selectedRoom = null;
            updateBookingButton();

            if (response.success && response.data.length > 0) {
                availableRooms = response.data;
                displayRooms();
                showToast(`Ditemukan ${response.data.length} kamar tersedia`, 'success');
            } else {
                showNoRoomsAvailable();
            }
        },
        error: function(xhr) {
            console.error('Room availability error:', xhr);
            let errorMsg = 'Terjadi kesalahan saat mengecek ketersediaan kamar';
            
            if (xhr.responseJSON && xhr.responseJSON.message) {
                errorMsg = xhr.responseJSON.message;
            }

            $('#roomsList').html(`
                <div class="empty-state">
                    <i class="fas fa-exclamation-triangle text-danger"></i>
                    <div>${errorMsg}</div>
                    <button class="btn btn-outline-primary btn-sm mt-2" onclick="loadAvailableRooms()">
                        <i class="fas fa-redo me-1"></i>Coba Lagi
                    </button>
                </div>
            `);
        }
    });
}

function displayRooms() {
    const nights = calculateNights();
    let html = '';

    if (availableRooms.length === 0) {
        showNoRoomsAvailable();
        return;
    }

    availableRooms.forEach((room, index) => {
        const totalPrice = room.price_per_night * nights;
        const ppn = Math.round(totalPrice * (room.ppn / 100));
        const grandTotal = totalPrice + ppn;

        html += `
            <div class="room-card fade-in-up" 
                 data-room-id="${room.id}" 
                 data-room-price="${room.price_per_night}" 
                 data-room-ppn="${room.ppn}"
                 data-room-number="${room.room_number}"
                 data-type-name="${room.type_name}"
                 style="animation-delay: ${index * 0.1}s">
                
                <div class="room-header">
                    <div class="room-details">
                        <h6>${escapeHtml(room.type_name)} - Kamar ${escapeHtml(room.room_number)}</h6>
                        <div class="room-description">
                            ${room.description ? escapeHtml(room.description) : 'Kamar dengan fasilitas lengkap'}
                        </div>
                        <div class="room-capacity">
                            <i class="fas fa-users"></i>
                            Max: ${room.max_capacity_adult} Dewasa, ${room.max_capacity_child} Anak
                        </div>
                    </div>
                    
                    <div class="room-pricing">
                        <div class="room-price">Rp ${grandTotal.toLocaleString('id-ID')}</div>
                        <div class="room-per-night">
                            ${nights} malam + PPN ${room.ppn}%<br>
                            <small>@ Rp ${room.price_per_night.toLocaleString('id-ID')}/malam</small>
                        </div>
                    </div>
                </div>
                
                <i class="fas fa-check-circle room-check-icon"></i>
            </div>
        `;
    });

    $('#roomsList').html(html);

    // Add click handlers for room selection
    $('.room-card').click(function() {
        selectRoom($(this));
    });
}

function selectRoom(element) {
    // Remove previous selection
    $('.room-card').removeClass('selected');
    element.addClass('selected bounce-in');
    
    selectedRoom = {
        id: element.data('room-id'),
        room_number: element.data('room-number'),
        type_name: element.data('type-name'),
        price_per_night: element.data('room-price'),
        ppn: element.data('room-ppn')
    };
    
    // Scroll to selected room
    element[0].scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    
    updateBookingButton();
    
    // Calculate and show price info
    const nights = calculateNights();
    const totalPrice = selectedRoom.price_per_night * nights;
    const ppn = Math.round(totalPrice * (selectedRoom.ppn / 100));
    const grandTotal = totalPrice + ppn;
    
    showToast(`Kamar dipilih: ${selectedRoom.type_name} - ${selectedRoom.room_number} | Total: Rp ${grandTotal.toLocaleString('id-ID')}`, 'success');
}

function showNoRoomsAvailable() {
    $('#roomsList').html(`
        <div class="empty-state">
            <i class="fas fa-bed text-muted"></i>
            <div>Tidak ada kamar tersedia pada tanggal yang dipilih</div>
            <small class="text-muted mt-2">Coba ubah tanggal atau kriteria pencarian</small>
        </div>
    `);
}

// Booking Confirmation & Processing
function confirmBooking() {
    // Validate all requirements
    if (!selectedGuest) {
        showToast('Silakan pilih tamu terlebih dahulu', 'error');
        return;
    }

    if (!selectedRoom) {
        showToast('Silakan pilih kamar terlebih dahulu', 'error');
        return;
    }

    const dateValidation = validateDates();
    if (!dateValidation.valid) {
        showToast(dateValidation.message, 'error');
        return;
    }

    // Additional validation for new guest
    if (selectedGuest.type === 'new' && !selectedGuest.name.trim()) {
        showToast('Nama tamu tidak boleh kosong', 'error');
        $('#newGuestName').focus();
        return;
    }

    // Calculate booking details
    const nights = calculateNights();
    const totalPrice = selectedRoom.price_per_night * nights;
    const ppn = Math.round(totalPrice * (selectedRoom.ppn / 100));
    const grandTotal = totalPrice + ppn;

    // Show confirmation dialog
    Swal.fire({
        title: 'Konfirmasi Booking',
        html: generateBookingSummary(nights, totalPrice, ppn, grandTotal),
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#28a745',
        cancelButtonColor: '#6c757d',
        confirmButtonText: '<i class="fas fa-check me-2"></i>Ya, Booking!',
        cancelButtonText: '<i class="fas fa-times me-2"></i>Batal',
        customClass: {
            confirmButton: 'btn btn-success',
            cancelButton: 'btn btn-secondary'
        }
    }).then((result) => {
        if (result.isConfirmed) {
            processBooking();
        }
    });
}

function generateBookingSummary(nights, totalPrice, ppn, grandTotal) {
    return `
        <div class="text-start">
            <div class="row mb-3">
                <div class="col-sm-4"><strong>Tamu:</strong></div>
                <div class="col-sm-8">${escapeHtml(selectedGuest.name)}</div>
            </div>
            <div class="row mb-3">
                <div class="col-sm-4"><strong>Kamar:</strong></div>
                <div class="col-sm-8">${escapeHtml(selectedRoom.type_name)} - ${escapeHtml(selectedRoom.room_number)}</div>
            </div>
            <div class="row mb-3">
                <div class="col-sm-4"><strong>Check-in:</strong></div>
                <div class="col-sm-8">${formatDate($('#checkInDate').val())}</div>
            </div>
            <div class="row mb-3">
                <div class="col-sm-4"><strong>Check-out:</strong></div>
                <div class="col-sm-8">${formatDate($('#checkOutDate').val())}</div>
            </div>
            <div class="row mb-3">
                <div class="col-sm-4"><strong>Durasi:</strong></div>
                <div class="col-sm-8">${nights} malam</div>
            </div>
            <hr>
            <div class="row mb-2">
                <div class="col-sm-8">Harga kamar (${nights} malam):</div>
                <div class="col-sm-4 text-end">Rp ${totalPrice.toLocaleString('id-ID')}</div>
            </div>
            <div class="row mb-2">
                <div class="col-sm-8">PPN (${selectedRoom.ppn}%):</div>
                <div class="col-sm-4 text-end">Rp ${ppn.toLocaleString('id-ID')}</div>
            </div>
            <div class="row">
                <div class="col-sm-8"><strong>Total Bayar:</strong></div>
                <div class="col-sm-4 text-end"><strong class="text-success">Rp ${grandTotal.toLocaleString('id-ID')}</strong></div>
            </div>
        </div>
    `;
}

function processBooking() {
    // Show loading state
    Swal.fire({
        title: 'Memproses Booking...',
        html: `
            <div class="text-center">
                <i class="fas fa-spinner fa-spin fa-2x text-primary mb-3"></i>
                <p>Mohon tunggu, sedang memproses booking Anda...</p>
            </div>
        `,
        allowOutsideClick: false,
        showConfirmButton: false,
        customClass: {
            popup: 'animated fadeIn'
        }
    });

    const bookingData = {
        guest_data: selectedGuest,
        room_id: selectedRoom.id,
        check_in_date: $('#checkInDate').val(),
        check_out_date: $('#checkOutDate').val()
    };

    $.ajax({
        url: '{{ route("bookings.store") }}',
        method: 'POST',
        data: bookingData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        },
        success: function(response) {
            if (response.success) {
                showBookingSuccess(response.data);
            } else {
                showBookingError('Booking gagal: ' + (response.message || 'Terjadi kesalahan'));
            }
        },
        error: function(xhr) {
            console.error('Booking error:', xhr);
            
            let errorMsg = 'Terjadi kesalahan saat memproses booking';
            let errorDetails = '';
            
            if (xhr.responseJSON) {
                if (xhr.status === 422) {
                    // Validation errors
                    const errors = xhr.responseJSON.errors;
                    errorDetails = '<ul class="text-start mt-2">';
                    Object.keys(errors).forEach(key => {
                        errorDetails += `<li>${errors[key][0]}</li>`;
                    });
                    errorDetails += '</ul>';
                } else {
                    errorMsg = xhr.responseJSON.message || errorMsg;
                }
            }
            
            showBookingError(errorMsg + errorDetails);
        }
    });
}

function showBookingSuccess(data) {
    Swal.fire({
        icon: 'success',
        title: 'Booking Berhasil!',
        html: `
            <div class="text-start">
                <div class="alert alert-success">
                    <i class="fas fa-check-circle me-2"></i>
                    Booking telah berhasil dibuat
                </div>
                <div class="booking-details">
                    <div class="row mb-2">
                        <div class="col-sm-5"><strong>ID Booking:</strong></div>
                        <div class="col-sm-7">#${data.booking_id}</div>
                    </div>
                    <div class="row mb-2">
                        <div class="col-sm-5"><strong>Tamu:</strong></div>
                        <div class="col-sm-7">${escapeHtml(data.guest_name)}</div>
                    </div>
                    <div class="row mb-2">
                        <div class="col-sm-5"><strong>Kamar:</strong></div>
                        <div class="col-sm-7">${escapeHtml(data.room_number)}</div>
                    </div>
                    <div class="row mb-2">
                        <div class="col-sm-5"><strong>Total:</strong></div>
                        <div class="col-sm-7"><strong class="text-success">Rp ${data.total_amount.toLocaleString('id-ID')}</strong></div>
                    </div>
                </div>
            </div>
        `,
        confirmButtonText: '<i class="fas fa-eye me-2"></i>Lihat Detail',
        showCancelButton: true,
        cancelButtonText: '<i class="fas fa-plus me-2"></i>Booking Lagi',
        customClass: {
            confirmButton: 'btn btn-primary',
            cancelButton: 'btn btn-success'
        }
    }).then((result) => {
        if (result.isConfirmed) {
            // Redirect to booking detail or list
            window.location.href = `{{ url('bookings') }}/${data.booking_id || ''}`;
        } else if (result.dismiss === Swal.DismissReason.cancel) {
            // Reset form for new booking
            resetForm();
        } else {
            // Close modal
            $('#quickBookingModal').modal('hide');
            
            // Reload page or update data
            if (typeof window.reloadBookings === 'function') {
                window.reloadBookings();
            } else {
                window.location.reload();
            }
        }
    });
}

function showBookingError(message) {
    Swal.fire({
        icon: 'error',
        title: 'Booking Gagal!',
        html: message,
        confirmButtonText: '<i class="fas fa-redo me-2"></i>Coba Lagi',
        customClass: {
            confirmButton: 'btn btn-danger'
        }
    });
}

// Utility Functions
function formatDate(dateString) {
    const options = { 
        weekday: 'long', 
        year: 'numeric', 
        month: 'long', 
        day: 'numeric' 
    };
    return new Date(dateString).toLocaleDateString('id-ID', options);
}

function formatCurrency(amount) {
    return new Intl.NumberFormat('id-ID', {
        style: 'currency',
        currency: 'IDR',
        minimumFractionDigits: 0
    }).format(amount);
}

// Add to global functions
window.quickBookingFunctions = {
    ...window.quickBookingFunctions,
    loadAvailableRooms,
    displayRooms,
    selectRoom,
    confirmBooking,
    processBooking,
    showBookingSuccess,
    showBookingError,
    formatDate,
    formatCurrency
};
</script>
@endpush